<?php
/**
 * USB Print Helper - Alternative printing solution for TSC USB printers
 * This file provides methods to print PRN files to USB printers when direct PHP methods fail
 */

function createPrintHelperScript($printerName, $prnFilePath) {
    // Create a PowerShell script that mimics the VB.NET UsbPrint functionality
    $psScript = "
# USB Printer Helper Script
param(
    [string]\$PrinterName = '$printerName',
    [string]\$PrnFile = '$prnFilePath'
)

try {
    # Method 1: Use .NET Framework classes (similar to VB.NET approach)
    Add-Type -AssemblyName System.Drawing
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing.Printing
    
    # Read PRN file content
    \$prnContent = Get-Content \$PrnFile -Raw -Encoding UTF8
    
    # Create PrintDocument object
    \$printDoc = New-Object System.Drawing.Printing.PrintDocument
    \$printDoc.PrinterSettings.PrinterName = \$PrinterName
    \$printDoc.DocumentName = 'TSC_Label_Print'
    
    # Set up the print event handler
    \$printHandler = {
        param(\$sender, \$e)
        
        # For TSC printers, we need to send raw data
        # Create a graphics object and send the raw PRN commands
        try {
            # Convert PRN content to bytes and send directly
            \$bytes = [System.Text.Encoding]::UTF8.GetBytes(\$prnContent)
            
            # Use RawPrinterHelper approach
            \$rawPrinter = @'
using System;
using System.IO;
using System.Runtime.InteropServices;

public class RawPrinterHelper {
    [StructLayout(LayoutKind.Sequential, CharSet=CharSet.Ansi)]
    public class DOCINFOA {
        [MarshalAs(UnmanagedType.LPStr)] public string pDocName;
        [MarshalAs(UnmanagedType.LPStr)] public string pOutputFile;
        [MarshalAs(UnmanagedType.LPStr)] public string pDataType;
    }

    [DllImport(\"winspool.Drv\", EntryPoint=\"OpenPrinterA\", SetLastError=true, CharSet=CharSet.Ansi, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool OpenPrinter([MarshalAs(UnmanagedType.LPStr)] string szPrinter, out IntPtr hPrinter, IntPtr pd);

    [DllImport(\"winspool.Drv\", EntryPoint=\"ClosePrinter\", SetLastError=true, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool ClosePrinter(IntPtr hPrinter);

    [DllImport(\"winspool.Drv\", EntryPoint=\"StartDocPrinterA\", SetLastError=true, CharSet=CharSet.Ansi, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool StartDocPrinter(IntPtr hPrinter, Int32 level, [In, MarshalAs(UnmanagedType.LPStruct)] DOCINFOA di);

    [DllImport(\"winspool.Drv\", EntryPoint=\"EndDocPrinter\", SetLastError=true, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool EndDocPrinter(IntPtr hPrinter);

    [DllImport(\"winspool.Drv\", EntryPoint=\"StartPagePrinter\", SetLastError=true, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool StartPagePrinter(IntPtr hPrinter);

    [DllImport(\"winspool.Drv\", EntryPoint=\"EndPagePrinter\", SetLastError=true, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool EndPagePrinter(IntPtr hPrinter);

    [DllImport(\"winspool.Drv\", EntryPoint=\"WritePrinter\", SetLastError=true, ExactSpelling=true, CallingConvention=CallingConvention.StdCall)]
    public static extern bool WritePrinter(IntPtr hPrinter, IntPtr pBytes, Int32 dwCount, out Int32 dwWritten);

    public static bool SendBytesToPrinter(string szPrinterName, IntPtr pBytes, Int32 dwCount) {
        Int32 dwError = 0, dwWritten = 0;
        IntPtr hPrinter = new IntPtr(0);
        DOCINFOA di = new DOCINFOA();
        bool bSuccess = false;

        di.pDocName = \"TSC Raw Document\";
        di.pDataType = \"RAW\";

        if (OpenPrinter(szPrinterName.Normalize(), out hPrinter, IntPtr.Zero)) {
            if (StartDocPrinter(hPrinter, 1, di)) {
                if (StartPagePrinter(hPrinter)) {
                    bSuccess = WritePrinter(hPrinter, pBytes, dwCount, out dwWritten);
                    EndPagePrinter(hPrinter);
                }
                EndDocPrinter(hPrinter);
            }
            ClosePrinter(hPrinter);
        }
        return bSuccess;
    }

    public static bool SendStringToPrinter(string szPrinterName, string szString) {
        IntPtr pBytes;
        Int32 dwCount;
        dwCount = szString.Length;
        pBytes = Marshal.StringToCoTaskMemAnsi(szString);
        bool result = SendBytesToPrinter(szPrinterName, pBytes, dwCount);
        Marshal.FreeCoTaskMem(pBytes);
        return result;
    }
}
'@

            Add-Type -TypeDefinition \$rawPrinter
            \$result = [RawPrinterHelper]::SendStringToPrinter(\$PrinterName, \$prnContent)
            
            if (\$result) {
                Write-Output \"SUCCESS: Raw data sent to USB printer successfully\"
            } else {
                Write-Output \"ERROR: Failed to send raw data to printer\"
            }
            
        } catch {
            Write-Output \"ERROR: \$(\$_.Exception.Message)\"
        }
        
        \$e.HasMorePages = \$false
    }
    
    # Add the event handler
    \$printDoc.add_PrintPage(\$printHandler)
    
    # Print the document
    \$printDoc.Print()
    
    Write-Output \"SUCCESS: Print job completed using .NET method\"
    
} catch {
    Write-Output \"ERROR: \$(\$_.Exception.Message)\"
    
    # Fallback method - try direct port communication
    try {
        \$printers = Get-WmiObject -Class Win32_Printer | Where-Object { \$_.Name -eq \$PrinterName }
        if (\$printers) {
            \$portName = \$printers[0].PortName
            if (\$portName) {
                Copy-Item \$PrnFile -Destination \$portName
                Write-Output \"SUCCESS: Fallback method - copied to port \$portName\"
            }
        }
    } catch {
        Write-Output \"ERROR: Fallback method failed - \$(\$_.Exception.Message)\"
    }
}
";

    return $psScript;
}

function printToUSBPrinter($printerName, $prnContent) {
    // Create temporary PRN file
    $tempFile = tempnam(sys_get_temp_dir(), 'usb_print_') . '.prn';
    file_put_contents($tempFile, $prnContent);
    
    // Create PowerShell script
    $script = createPrintHelperScript($printerName, $tempFile);
    $scriptFile = tempnam(sys_get_temp_dir(), 'ps_print_') . '.ps1';
    file_put_contents($scriptFile, $script);
    
    // Execute PowerShell script
    $command = "powershell -ExecutionPolicy Bypass -File \"$scriptFile\" 2>&1";
    $output = [];
    $returnVar = 0;
    exec($command, $output, $returnVar);
    
    $outputStr = implode(' ', $output);
    $success = (strpos($outputStr, 'SUCCESS:') !== false);
    
    // Clean up temporary files
    if (file_exists($tempFile)) unlink($tempFile);
    if (file_exists($scriptFile)) unlink($scriptFile);
    
    return [
        'success' => $success,
        'output' => $outputStr,
        'return_code' => $returnVar
    ];
}

// Test function to verify printer connectivity
function testUSBPrinter($printerName) {
    $testScript = "
        try {
            \$printer = Get-Printer -Name '$printerName' -ErrorAction Stop
            if (\$printer.PrinterStatus -eq 'Normal') {
                Write-Output 'SUCCESS: Printer is ready and accessible'
            } else {
                Write-Output 'WARNING: Printer status is: ' + \$printer.PrinterStatus
            }
        } catch {
            Write-Output 'ERROR: Printer not found or not accessible'
        }
    ";
    
    $scriptFile = tempnam(sys_get_temp_dir(), 'test_printer_') . '.ps1';
    file_put_contents($scriptFile, $testScript);
    
    $command = "powershell -ExecutionPolicy Bypass -File \"$scriptFile\" 2>&1";
    $output = [];
    exec($command, $output, $returnVar);
    
    $outputStr = implode(' ', $output);
    $success = (strpos($outputStr, 'SUCCESS:') !== false);
    
    if (file_exists($scriptFile)) unlink($scriptFile);
    
    return [
        'success' => $success,
        'output' => $outputStr
    ];
}
?>
