<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];
$msg = "";

// Initialize filter variables
$dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$projectFilter = isset($_GET['project']) ? mysqli_real_escape_string($con, $_GET['project']) : '';
$statusFilter = isset($_GET['status']) ? $_GET['status'] : 'all';

// Build WHERE conditions
$whereConditions = [];
if (!empty($projectFilter)) {
    $whereConditions[] = "i.ProjectName LIKE '%$projectFilter%'";
}

$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Main query for project-wise detailed report
$projectReportQuery = "
    SELECT 
        ProjectName,
        SUM(TotalItems) as TotalItems,
        SUM(TotalInventoryQty) as TotalInventoryQty,
        SUM(TotalInventoryWeight) as TotalInventoryWeight,
        SUM(TotalPackedQty) as TotalPackedQty,
        SUM(TotalPackedWeight) as TotalPackedWeight,
        SUM(RemainingQty) as RemainingQty,
        SUM(RemainingWeight) as RemainingWeight,
        ROUND((SUM(TotalPackedQty) / SUM(TotalInventoryQty) * 100), 2) as CompletionPercentage,
        CASE 
            WHEN SUM(RemainingQty) = 0 THEN 'Closed'
            WHEN SUM(TotalPackedQty) > 0 THEN 'Active'
            ELSE 'Pending'
        END as ProjectStatus,
        MIN(ProjectStartDate) as ProjectStartDate,
        MAX(LastActivity) as LastActivity,
        SUM(TotalPackages) as TotalPackages,
        SUM(TotalRooms) as TotalRooms
    FROM (
        SELECT 
            i.ProjectName,
            1 as TotalItems,
            i.Qty as TotalInventoryQty,
            i.Weight as TotalInventoryWeight,
            COALESCE(SUM(p.PackedQty), 0) as TotalPackedQty,
            COALESCE(SUM(p.PackedWeight), 0) as TotalPackedWeight,
            (i.Qty - COALESCE(SUM(p.PackedQty), 0)) as RemainingQty,
            (i.Weight - COALESCE(SUM(p.PackedWeight), 0)) as RemainingWeight,
            i.UploadDate as ProjectStartDate,
            COALESCE(MAX(p.PackedDate), i.UploadDate) as LastActivity,
            COUNT(DISTINCT p.PackageId) as TotalPackages,
            1 as TotalRooms
        FROM tblinventory i
        LEFT JOIN tblpacked p ON i.ID = p.ProductId
        $whereClause
        GROUP BY i.ID, i.ProjectName, i.Qty, i.Weight, i.UploadDate, i.RoomNo
    ) as item_status
    GROUP BY ProjectName
";

// Apply status filter after grouping
if ($statusFilter !== 'all') {
    $projectReportQuery = "SELECT * FROM ($projectReportQuery) as filtered_data WHERE ";
    switch($statusFilter) {
        case 'pending':
            $projectReportQuery .= "ProjectStatus = 'Pending'";
            break;
        case 'active':
            $projectReportQuery .= "ProjectStatus = 'Active'";
            break;
        case 'closed':
            $projectReportQuery .= "ProjectStatus = 'Closed'";
            break;
    }
}

$projectReportQuery .= " ORDER BY ProjectStatus DESC, CompletionPercentage DESC";

$projectReportResult = mysqli_query($con, $projectReportQuery);

// Get overall summary
$summaryQuery = "
    SELECT 
        COUNT(*) as TotalProjects,
        SUM(TotalInventoryQty) as GrandTotalInventory,
        SUM(TotalInventoryWeight) as GrandTotalWeight,
        SUM(TotalPackedQty) as GrandTotalPacked,
        SUM(TotalPackedWeight) as GrandTotalPackedWeight,
        SUM(RemainingQty) as GrandTotalRemaining,
        SUM(RemainingWeight) as GrandTotalRemainingWeight,
        SUM(CASE WHEN ProjectStatus = 'Pending' THEN 1 ELSE 0 END) as PendingCount,
        SUM(CASE WHEN ProjectStatus = 'Active' THEN 1 ELSE 0 END) as ActiveCount,
        SUM(CASE WHEN ProjectStatus = 'Closed' THEN 1 ELSE 0 END) as ClosedCount
    FROM ($projectReportQuery) as summary_data
";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Get filter dropdown data
$projectsQuery = mysqli_query($con, "SELECT DISTINCT ProjectName FROM tblinventory WHERE ProjectName != '' ORDER BY ProjectName");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Project Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .status-closed { background-color: #d4edda; color: #155724; }
        .status-active { background-color: #fff3cd; color: #856404; }
        .status-pending { background-color: #f8d7da; color: #721c24; }
        .filter-section { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .summary-card { transition: transform 0.2s; }
        .summary-card:hover { transform: translateY(-2px); }
        .progress-thin { height: 8px; }
        @media print {
            .no-print { display: none !important; }
            .card { box-shadow: none !important; border: 1px solid #ddd !important; }
        }
        @media (max-width: 768px) {
            .table-responsive table, .table-responsive thead, .table-responsive tbody, 
            .table-responsive th, .table-responsive td, .table-responsive tr {
                display: block;
            }
            .table-responsive thead tr { position: absolute; top: -9999px; left: -9999px; }
            .table-responsive tr { border: 1px solid #ccc; margin-bottom: 10px; }
            .table-responsive td {
                border: none; position: relative; padding-left: 50% !important;
            }
            .table-responsive td:before {
                content: attr(data-label) ": ";
                position: absolute; left: 6px; width: 45%; padding-right: 10px;
                white-space: nowrap; font-weight: bold;
            }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                
                <!-- Begin Page Content -->
                <div class="container-fluid">
                    
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">
                            <i class="fas fa-project-diagram text-primary mr-2"></i>
                            Project Status Report
                        </h1>
                        <div class="no-print">
                            <button onclick="window.print()" class="btn btn-primary btn-sm">
                                <i class="fas fa-print mr-2"></i>Print Report
                            </button>
                            <button onclick="exportToExcel()" class="btn btn-success btn-sm">
                                <i class="fas fa-file-excel mr-2"></i>Export Excel
                            </button>
                        </div>
                    </div>

                    <!-- Filter Section -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header filter-section text-white">
                            <div class="row align-items-center">
                                <div class="col">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-filter mr-2"></i>Report Filters
                                    </h6>
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <form method="GET" action="">
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="project">Project Name</label>
                                            <select class="form-control" id="project" name="project">
                                                <option value="">All Projects</option>
                                                <?php while($project = mysqli_fetch_assoc($projectsQuery)): ?>
                                                <option value="<?php echo htmlspecialchars($project['ProjectName']); ?>" 
                                                    <?php echo ($projectFilter == $project['ProjectName']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($project['ProjectName']); ?>
                                                </option>
                                                <?php endwhile; ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="status">Project Status</label>
                                            <select class="form-control" id="status" name="status">
                                                <option value="all" <?php echo ($statusFilter == 'all') ? 'selected' : ''; ?>>All Status</option>
                                                <option value="pending" <?php echo ($statusFilter == 'pending') ? 'selected' : ''; ?>>Pending</option>
                                                <option value="active" <?php echo ($statusFilter == 'active') ? 'selected' : ''; ?>>Active</option>
                                                <option value="closed" <?php echo ($statusFilter == 'closed') ? 'selected' : ''; ?>>Closed</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div class="d-flex">
                                                <button type="submit" class="btn btn-primary mr-2">
                                                    <i class="fas fa-search mr-1"></i>Apply Filters
                                                </button>
                                                <a href="project-report.php" class="btn btn-secondary">
                                                    <i class="fas fa-times mr-1"></i>Clear
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row">
                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-primary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                                Total Projects</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['TotalProjects']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-project-diagram fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-success shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                                Closed</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['ClosedCount']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-warning shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                                Active</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['ActiveCount']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-play-circle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-danger shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                                Pending</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['PendingCount']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-pause-circle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-info shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                                Total Inventory</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['GrandTotalInventory']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-boxes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-2 col-md-6 mb-4">
                            <div class="card summary-card border-left-secondary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-secondary text-uppercase mb-1">
                                                Remaining</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo number_format($summary['GrandTotalRemaining']); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-warehouse fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Project Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <i class="fas fa-table mr-2"></i>
                                Project Details Report
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-striped" id="projectReportTable" width="100%" cellspacing="0">
                                    <thead class="thead-dark">
                                        <tr>
                                            <th>Project Name</th>
                                            <th>Status</th>
                                            <th>Progress</th>
                                            <th>Items</th>
                                            <th>Inventory Qty</th>
                                            <th>Inventory Weight (kg)</th>
                                            <th>Packed Qty</th>
                                            <th>Packed Weight (kg)</th>
                                            <th>Remaining Qty</th>
                                            <th>Remaining Weight (kg)</th>
                                            <th>Packages</th>
                                            <th>Rooms</th>
                                            <th>Start Date</th>
                                            <th>Last Activity</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        $totalInventoryQty = 0;
                                        $totalInventoryWeight = 0;
                                        $totalPackedQty = 0;
                                        $totalPackedWeight = 0;
                                        $totalRemainingQty = 0;
                                        $totalRemainingWeight = 0;
                                        
                                        while($project = mysqli_fetch_assoc($projectReportResult)): 
                                            $totalInventoryQty += $project['TotalInventoryQty'];
                                            $totalInventoryWeight += $project['TotalInventoryWeight'];
                                            $totalPackedQty += $project['TotalPackedQty'];
                                            $totalPackedWeight += $project['TotalPackedWeight'];
                                            $totalRemainingQty += $project['RemainingQty'];
                                            $totalRemainingWeight += $project['RemainingWeight'];
                                        ?>
                                        <tr>
                                            <td data-label="Project Name">
                                                <strong><?php echo htmlspecialchars($project['ProjectName']); ?></strong>
                                            </td>
                                            <td data-label="Status">
                                                <span class="badge badge-pill px-3 py-2 status-<?php echo strtolower($project['ProjectStatus']); ?>">
                                                    <?php echo $project['ProjectStatus']; ?>
                                                </span>
                                            </td>
                                            <td data-label="Progress">
                                                <div class="mb-1">
                                                    <small class="text-muted"><?php echo $project['CompletionPercentage']; ?>%</small>
                                                </div>
                                                <div class="progress progress-thin">
                                                    <div class="progress-bar 
                                                        <?php 
                                                        if($project['CompletionPercentage'] == 100) echo 'bg-success';
                                                        elseif($project['CompletionPercentage'] >= 50) echo 'bg-warning'; 
                                                        else echo 'bg-danger';
                                                        ?>" 
                                                        role="progressbar" 
                                                        style="width: <?php echo $project['CompletionPercentage']; ?>%">
                                                    </div>
                                                </div>
                                            </td>
                                            <td data-label="Items"><?php echo number_format($project['TotalItems']); ?></td>
                                            <td data-label="Inventory Qty"><?php echo number_format($project['TotalInventoryQty']); ?></td>
                                            <td data-label="Inventory Weight"><?php echo number_format($project['TotalInventoryWeight'], 2); ?></td>
                                            <td data-label="Packed Qty" class="text-success font-weight-bold"><?php echo number_format($project['TotalPackedQty']); ?></td>
                                            <td data-label="Packed Weight" class="text-success"><?php echo number_format($project['TotalPackedWeight'], 2); ?></td>
                                            <td data-label="Remaining Qty" class="<?php echo $project['RemainingQty'] > 0 ? 'text-warning font-weight-bold' : 'text-success'; ?>">
                                                <?php echo number_format($project['RemainingQty']); ?>
                                            </td>
                                            <td data-label="Remaining Weight" class="<?php echo $project['RemainingWeight'] > 0 ? 'text-warning' : 'text-success'; ?>">
                                                <?php echo number_format($project['RemainingWeight'], 2); ?>
                                            </td>
                                            <td data-label="Packages"><?php echo number_format($project['TotalPackages']); ?></td>
                                            <td data-label="Rooms"><?php echo number_format($project['TotalRooms']); ?></td>
                                            <td data-label="Start Date"><?php echo date('M j, Y', strtotime($project['ProjectStartDate'])); ?></td>
                                            <td data-label="Last Activity"><?php echo date('M j, Y', strtotime($project['LastActivity'])); ?></td>
                                        </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                    <tfoot class="thead-light">
                                        <tr>
                                            <th colspan="4"><strong>TOTAL</strong></th>
                                            <th><strong><?php echo number_format($totalInventoryQty); ?></strong></th>
                                            <th><strong><?php echo number_format($totalInventoryWeight, 2); ?></strong></th>
                                            <th class="text-success"><strong><?php echo number_format($totalPackedQty); ?></strong></th>
                                            <th class="text-success"><strong><?php echo number_format($totalPackedWeight, 2); ?></strong></th>
                                            <th class="text-warning"><strong><?php echo number_format($totalRemainingQty); ?></strong></th>
                                            <th class="text-warning"><strong><?php echo number_format($totalRemainingWeight, 2); ?></strong></th>
                                            <th colspan="4"></th>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                        </div>
                    </div>

                </div>
                <!-- /.container-fluid -->
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>

    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    
    <script>
        function exportToExcel() {
            const wb = XLSX.utils.book_new();
            
            // PROJECT SUMMARY SHEET
            const summaryData = [
                ['PROJECT STATUS REPORT'],
                ['Generated:', new Date().toLocaleDateString('en-GB') + ' ' + new Date().toLocaleTimeString()],
                [''],
                ['EXECUTIVE SUMMARY'],
                ['Total Projects', '<?php echo number_format($summary['TotalProjects']); ?>'],
                ['Closed Projects', '<?php echo number_format($summary['ClosedCount']); ?>'],
                ['Active Projects', '<?php echo number_format($summary['ActiveCount']); ?>'],
                ['Pending Projects', '<?php echo number_format($summary['PendingCount']); ?>'],
                [''],
                ['INVENTORY OVERVIEW'],
                ['Total Inventory Quantity', '<?php echo number_format($summary['GrandTotalInventory']); ?>'],
                ['Total Inventory Weight (kg)', '<?php echo number_format($summary['GrandTotalWeight'], 2); ?>'],
                ['Total Packed Quantity', '<?php echo number_format($summary['GrandTotalPacked']); ?>'],
                ['Total Packed Weight (kg)', '<?php echo number_format($summary['GrandTotalPackedWeight'], 2); ?>'],
                ['Total Remaining Quantity', '<?php echo number_format($summary['GrandTotalRemaining']); ?>'],
                ['Total Remaining Weight (kg)', '<?php echo number_format($summary['GrandTotalRemainingWeight'], 2); ?>'],
                [''],
                ['PERFORMANCE METRICS'],
                ['Overall Completion Rate', '<?php echo $summary['GrandTotalInventory'] > 0 ? round(($summary['GrandTotalPacked'] / $summary['GrandTotalInventory']) * 100, 2) : 0; ?>%'],
                ['Projects Completion Rate', '<?php echo $summary['TotalProjects'] > 0 ? round(($summary['ClosedCount'] / $summary['TotalProjects']) * 100, 2) : 0; ?>%']
            ];
            
            const summaryWS = XLSX.utils.aoa_to_sheet(summaryData);
            summaryWS['!cols'] = [{ width: 30 }, { width: 20 }];
            
            XLSX.utils.book_append_sheet(wb, summaryWS, 'Executive Summary');
            
            // DETAILED PROJECT REPORT SHEET
            const table = document.getElementById('projectReportTable');
            const detailedWS = XLSX.utils.table_to_sheet(table);
            
            // Set column widths for detailed sheet
            detailedWS['!cols'] = [
                { width: 35 }, // Project Name
                { width: 12 }, // Status
                { width: 10 }, // Progress
                { width: 8 },  // Items
                { width: 12 }, // Inventory Qty
                { width: 15 }, // Inventory Weight
                { width: 12 }, // Packed Qty
                { width: 15 }, // Packed Weight
                { width: 12 }, // Remaining Qty
                { width: 15 }, // Remaining Weight
                { width: 10 }, // Packages
                { width: 8 },  // Rooms
                { width: 12 }, // Start Date
                { width: 12 }  // Last Activity
            ];
            
            XLSX.utils.book_append_sheet(wb, detailedWS, 'Detailed Project Report');
            
            // Save file
            const fileName = 'Project_Status_Report_' + new Date().toISOString().slice(0, 10) + '_' + 
                           new Date().toTimeString().slice(0, 5).replace(':', '') + '.xlsx';
            XLSX.writeFile(wb, fileName);
        }
    </script>
</body>
</html>
