<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];
$msg = "";

// Initialize filter variables
$dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$projectFilter = isset($_GET['project']) ? mysqli_real_escape_string($con, $_GET['project']) : '';
$roomFilter = isset($_GET['room']) ? mysqli_real_escape_string($con, $_GET['room']) : '';
$productFilter = isset($_GET['product']) ? mysqli_real_escape_string($con, $_GET['product']) : '';
$packageFilter = isset($_GET['package']) ? mysqli_real_escape_string($con, $_GET['package']) : '';
$userFilter = isset($_GET['user']) ? mysqli_real_escape_string($con, $_GET['user']) : '';

// Build WHERE conditions
$whereConditions = [];
$whereConditions[] = "p.PackedDate BETWEEN '$dateFrom 00:00:00' AND '$dateTo 23:59:59'";

if (!empty($projectFilter)) {
    $whereConditions[] = "p.ProjectName LIKE '%$projectFilter%'";
}
if (!empty($roomFilter)) {
    $whereConditions[] = "p.RoomNo LIKE '%$roomFilter%'";
}
if (!empty($productFilter)) {
    $whereConditions[] = "p.ProductName LIKE '%$productFilter%'";
}
if (!empty($packageFilter)) {
    $whereConditions[] = "(p.PackageId LIKE '%$packageFilter%' OR p.PackageName LIKE '%$packageFilter%')";
}
if (!empty($userFilter) && $userFilter !== 'all') {
    $whereConditions[] = "p.PackedBy = '$userFilter'";
}

$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Main query to get packed items
$reportQuery = "
    SELECT 
        p.ID,
        p.PackageId,
        p.PackageName,
        p.Barcode,
        p.ProductName,
        p.RoomNo,
        p.ProjectName,
        p.PackedQty,
        p.PackedWeight,
        p.PackedDate,
        u.FullName as PackedBy,
        i.Qty as OriginalQty,
        i.Weight as OriginalWeight,
        CASE 
            WHEN i.Qty IS NULL THEN 'Product Not Found'
            WHEN p.PackedQty > i.Qty THEN 'Over Packed'
            ELSE 'Normal'
        END as PackingStatus
    FROM tblpacked p
    LEFT JOIN tblusers u ON p.PackedBy = u.ID
    LEFT JOIN tblinventory i ON p.ProductId = i.ID
    $whereClause
    ORDER BY p.PackedDate ASC, p.PackageName ASC
";

$reportResult = mysqli_query($con, $reportQuery);

// Get summary statistics
$summaryQuery = "
    SELECT 
        COUNT(*) as TotalPackedItems,
        COUNT(DISTINCT PackageId) as TotalPackages,
        COALESCE(SUM(PackedQty), 0) as TotalPackedQty,
        COALESCE(SUM(PackedWeight), 0) as TotalPackedWeight,
        COUNT(DISTINCT ProjectName) as TotalProjects,
        COUNT(DISTINCT RoomNo) as TotalRooms,
        COUNT(DISTINCT PackedBy) as TotalPackers
    FROM tblpacked p
    $whereClause
";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Handle null values from summary query
$summary['TotalPackedItems'] = $summary['TotalPackedItems'] ?? 0;
$summary['TotalPackages'] = $summary['TotalPackages'] ?? 0;
$summary['TotalPackedQty'] = $summary['TotalPackedQty'] ?? 0;
$summary['TotalPackedWeight'] = $summary['TotalPackedWeight'] ?? 0;
$summary['TotalProjects'] = $summary['TotalProjects'] ?? 0;
$summary['TotalRooms'] = $summary['TotalRooms'] ?? 0;
$summary['TotalPackers'] = $summary['TotalPackers'] ?? 0;

// Get filter dropdown data
$projectsQuery = mysqli_query($con, "SELECT DISTINCT ProjectName FROM tblpacked WHERE ProjectName != '' ORDER BY ProjectName");
$roomsQuery = mysqli_query($con, "SELECT DISTINCT RoomNo FROM tblpacked WHERE RoomNo != '' ORDER BY RoomNo");
$packagesQuery = mysqli_query($con, "SELECT DISTINCT PackageId, PackageName FROM tblpacked ORDER BY PackageName");
$usersQuery = mysqli_query($con, "SELECT ID, FullName FROM tblusers ORDER BY FullName");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Packed Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .status-normal { background-color: #d4edda; color: #155724; }
        .status-over-packed { background-color: #f8d7da; color: #721c24; }
        .status-product-not-found { background-color: #fff3cd; color: #856404; }
        .filter-section { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .summary-card { transition: transform 0.2s; }
        .summary-card:hover { transform: translateY(-2px); }
        @media print {
            .no-print { display: none !important; }
            .card { box-shadow: none !important; border: 1px solid #ddd !important; }
        }
        @media (max-width: 768px) {
            .table-responsive table, .table-responsive thead, .table-responsive tbody, 
            .table-responsive th, .table-responsive td, .table-responsive tr {
                display: block;
            }
            .table-responsive thead tr { position: absolute; top: -9999px; left: -9999px; }
            .table-responsive tr { border: 1px solid #ccc; margin-bottom: 10px; }
            .table-responsive td {
                border: none; position: relative; padding-left: 50% !important;
            }
            .table-responsive td:before {
                content: attr(data-label) ": "; position: absolute; left: 6px; width: 45%;
                padding-right: 10px; white-space: nowrap; font-weight: bold;
            }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-3 mb-3">
                    
                    <!-- Filter Section -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header filter-section text-white">
                            <h5 class="mb-0"><i class="fas fa-filter"></i> Packed Report Filters</h5>
                        </div>
                        <div class="card-body">
                            <form method="GET" action="">
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date From</label>
                                            <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date To</label>
                                            <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Project</label>
                                            <select name="project" class="form-control">
                                                <option value="">All Projects</option>
                                                <?php while($project = mysqli_fetch_assoc($projectsQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($project['ProjectName']); ?>" 
                                                            <?php echo ($projectFilter == $project['ProjectName']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($project['ProjectName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Room</label>
                                            <select name="room" class="form-control">
                                                <option value="">All Rooms</option>
                                                <?php while($room = mysqli_fetch_assoc($roomsQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($room['RoomNo']); ?>" 
                                                            <?php echo ($roomFilter == $room['RoomNo']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($room['RoomNo']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Product Name</label>
                                            <input type="text" name="product" class="form-control" placeholder="Search product..." value="<?php echo htmlspecialchars($productFilter); ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Bundle</label>
                                            <select name="package" class="form-control">
                                                <option value="">All Bundles</option>
                                                <?php while($package = mysqli_fetch_assoc($packagesQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($package['PackageId']); ?>" 
                                                            <?php echo ($packageFilter == $package['PackageId']) ? 'selected' : ''; ?>>
                                                        Bundle #<?php echo htmlspecialchars($package['PackageName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Packed By</label>
                                            <select name="user" class="form-control">
                                                <option value="">All Users</option>
                                                <?php while($user = mysqli_fetch_assoc($usersQuery)) { ?>
                                                    <option value="<?php echo $user['ID']; ?>" 
                                                            <?php echo ($userFilter == $user['ID']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($user['FullName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="fas fa-search"></i> Filter
                                                </button>
                                                <a href="packed-report.php" class="btn btn-secondary">
                                                    <i class="fas fa-undo"></i> Reset
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-primary shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Packed Items</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackedItems']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-boxes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-success shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Bundles</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackages']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-box fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-warning shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Total Quantity</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackedQty']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-cubes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-info shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Total Weight</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackedWeight'], 2); ?> kg</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-weight fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-info shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Projects</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalProjects']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-building fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-secondary shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-secondary text-uppercase mb-1">Rooms</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalRooms']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-door-open fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-dark shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-dark text-uppercase mb-1">Packers</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackers']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-users fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header bg-primary text-white">
                            <div class="row align-items-center">
                                <div class="col">
                                    <h5 class="mb-0"><i class="fas fa-chart-bar"></i> Packed Items Report</h5>
                                </div>
                                <div class="col-auto no-print">
                                    <button onclick="window.print()" class="btn btn-light btn-sm">
                                        <i class="fas fa-print"></i> Print Report
                                    </button>
                                    <button onclick="exportToCSV()" class="btn btn-success btn-sm">
                                        <i class="fas fa-file-csv"></i> Export Packing List CSV
                                    </button>
                                    <a href="print-packing-list.php?<?php echo http_build_query($_GET); ?>" target="_blank" class="btn btn-warning btn-sm">
                                        <i class="fas fa-print"></i> Print Packing List
                                    </a>
                                </div>
                            </div>
                        </div>
                        <div class="card-body p-2">
                            <?php if (mysqli_num_rows($reportResult) == 0) { ?>
                                <div class="text-center text-muted py-4">
                                    <i class="fas fa-search fa-3x mb-3"></i>
                                    <h5>No data found</h5>
                                    <p>Try adjusting your filter criteria</p>
                                </div>
                            <?php } else { ?>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-sm mb-0" id="reportTable">
                                        <thead class="thead-light">
                                            <tr>
                                                <th>#</th>
                                                <th>Bundle Number</th>
                                                <th>Barcode</th>
                                                <th>Product Name</th>
                                                <th>Project</th>
                                                <th>Room</th>
                                                <th>Packed Qty</th>
                                                <th>Packed Weight</th>
                                                <th>Original Qty</th>
                                                <th>Original Weight</th>
                                                <th>Status</th>
                                                <th>Packed Date</th>
                                                <th>Packed By</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php $i=1; while($row = mysqli_fetch_assoc($reportResult)) { ?>
                                                <tr class="<?php 
                                                    echo $row['PackingStatus'] == 'Normal' ? 'status-normal' : 
                                                        ($row['PackingStatus'] == 'Over Packed' ? 'status-over-packed' : 'status-product-not-found'); 
                                                ?>">
                                                    <td data-label="#"><?php echo $i++; ?></td>
                                                    <td data-label="Bundle Number"><?php echo htmlspecialchars($row['PackageName']); ?></td>
                                                    <td data-label="Barcode"><?php echo htmlspecialchars($row['Barcode']); ?></td>
                                                    <td data-label="Product Name"><?php echo htmlspecialchars($row['ProductName']); ?></td>
                                                    <td data-label="Project"><?php echo htmlspecialchars($row['ProjectName']); ?></td>
                                                    <td data-label="Room"><?php echo htmlspecialchars($row['RoomNo']); ?></td>
                                                    <td data-label="Packed Qty"><strong class="text-success"><?php echo number_format($row['PackedQty']); ?></strong></td>
                                                    <td data-label="Packed Weight"><strong class="text-info"><?php echo number_format($row['PackedWeight'], 2); ?> kg</strong></td>
                                                    <td data-label="Original Qty"><?php echo $row['OriginalQty'] ? number_format($row['OriginalQty']) : 'N/A'; ?></td>
                                                    <td data-label="Original Weight"><?php echo $row['OriginalWeight'] ? number_format($row['OriginalWeight'], 2) . ' kg' : 'N/A'; ?></td>
                                                    <td data-label="Status">
                                                        <span class="badge badge-<?php 
                                                            echo $row['PackingStatus'] == 'Normal' ? 'success' : 
                                                                ($row['PackingStatus'] == 'Over Packed' ? 'danger' : 'warning'); 
                                                        ?>"><?php echo $row['PackingStatus']; ?></span>
                                                    </td>
                                                    <td data-label="Packed Date"><?php echo date('d-m-Y H:i', strtotime($row['PackedDate'])); ?></td>
                                                    <td data-label="Packed By"><?php echo htmlspecialchars($row['PackedBy']); ?></td>
                                                </tr>
                                            <?php } ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php } ?>
                        </div>
                    </div>

                    <div class="row no-print">
                        <div class="col-md-6">
                            <a href="dashboard.php" class="btn btn-secondary btn-block">
                                <i class="fas fa-arrow-left"></i> Back to Dashboard
                            </a>
                        </div>
                        <div class="col-md-6">
                            <a href="packing-module.php" class="btn btn-primary btn-block">
                                <i class="fas fa-plus"></i> Pack Items
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    
    <script>
        function exportToCSV() {
            // Get the same data that would be shown in print-packing-list.php
            const currentUrl = new URL(window.location.href);
            const params = new URLSearchParams(currentUrl.search);
            
            // Make AJAX request to get the packing list data
            fetch('get-packing-list-data.php?' + params.toString())
                .then(response => response.json())
                .then(data => {
                    let csvContent = '';
                    
                    // Header matching print-packing-list.php format
                    const projectTitle = data.projectTitle || 'Packing List';
                    csvContent += projectTitle + '\n';
                    
                    let subtitle = '';
                    if (data.projectName) {
                        subtitle += 'Project: ' + data.projectName + ' | ';
                    }
                    subtitle += 'Date: <?php echo date('d-m-Y', strtotime($dateFrom)); ?> to <?php echo date('d-m-Y', strtotime($dateTo)); ?>';
                    csvContent += subtitle + '\n\n';
                    
                    // CSV Headers matching print-packing-list.php table structure
                    csvContent += 'Sl No,Barcode,Description,Qty,Weight,Room,Bundle No,Date,Packed By,Status\n';
                    
                    // Data rows
                    let totalQty = 0;
                    let totalWeight = 0;
                    
                    data.rows.forEach((row, index) => {
                        const csvRow = [
                            index + 1, // Sl No
                            row.Barcode || '', // Barcode
                            '"' + row.ProductName.replace(/"/g, '""') + '"', // Description
                            row.PackedQty, // Qty
                            parseFloat(row.PackedWeight).toFixed(2), // Weight
                            row.RoomNo, // Room
                            row.PackageName, // Bundle No
                            new Date(row.PackedDate).toLocaleDateString('en-GB'), // Date
                            row.PackedBy, // Packed By
                            'Packed' // Status
                        ];
                        
                        csvContent += csvRow.join(',') + '\n';
                        totalQty += parseInt(row.PackedQty) || 0;
                        totalWeight += parseFloat(row.PackedWeight) || 0;
                    });
                    
                    // Total row
                    csvContent += '\n';
                    csvContent += 'Total,,,';
                    csvContent += totalQty + ',';
                    csvContent += totalWeight.toFixed(2) + ',,,,,\n';
                    
                    // Download the CSV file
                    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
                    const url = window.URL.createObjectURL(blob);
                    const link = document.createElement('a');
                    
                    const today = new Date();
                    const dateStr = today.getDate().toString().padStart(2, '0') + '-' + 
                                   (today.getMonth() + 1).toString().padStart(2, '0') + '-' + 
                                   today.getFullYear();
                    
                    link.href = url;
                    link.download = 'Packing_List_' + dateStr + '.csv';
                    link.style.display = 'none';
                    
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                    
                    window.URL.revokeObjectURL(url);
                })
                .catch(error => {
                    console.error('Error fetching packing list data:', error);
                    alert('Error generating CSV file: ' + error.message);
                });
        }
    </script>
</body>
</html>
