<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];
$msg = "";

// Initialize filter variables
$dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
$dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$projectFilter = isset($_GET['project']) ? mysqli_real_escape_string($con, $_GET['project']) : '';
$bundleFilter = isset($_GET['bundle']) ? mysqli_real_escape_string($con, $_GET['bundle']) : '';
$userFilter = isset($_GET['user']) ? mysqli_real_escape_string($con, $_GET['user']) : '';

// Debug: Check database connection and table status
$debug_info = "";
if (isset($_GET['debug'])) {
    // Check if tblpacked table exists and has data
    $checkTable = mysqli_query($con, "SHOW TABLES LIKE 'tblpacked'");
    $tableExists = mysqli_num_rows($checkTable) > 0;
    
    $packedCount = 0;
    if ($tableExists) {
        $countQuery = mysqli_query($con, "SELECT COUNT(*) as total FROM tblpacked");
        if ($countQuery) {
            $countRow = mysqli_fetch_assoc($countQuery);
            $packedCount = $countRow['total'];
        }
    }
    
    $inventoryCount = 0;
    $invCountQuery = mysqli_query($con, "SELECT COUNT(*) as total FROM tblinventory");
    if ($invCountQuery) {
        $invCountRow = mysqli_fetch_assoc($invCountQuery);
        $inventoryCount = $invCountRow['total'];
    }
    
    $debug_info = "Debug Info: Table exists: " . ($tableExists ? "Yes" : "No") . 
                  ", Packed records: $packedCount, Inventory records: $inventoryCount";
}

// Handle delete request
if (isset($_GET['delete']) && !empty($_GET['delete'])) {
    $packageId = mysqli_real_escape_string($con, $_GET['delete']);
    
    // Start transaction
    mysqli_autocommit($con, FALSE);
    
    try {
        // First, get the project name and package number before deletion
        $getPackageInfoQuery = "SELECT DISTINCT ProjectName, PackageName FROM tblpacked WHERE PackageId = '$packageId' LIMIT 1";
        $packageInfoResult = mysqli_query($con, $getPackageInfoQuery);
        
        if (!$packageInfoResult || mysqli_num_rows($packageInfoResult) == 0) {
            throw new Exception("Package not found");
        }
        
        $packageInfo = mysqli_fetch_assoc($packageInfoResult);
        $projectName = $packageInfo['ProjectName'];
        $packageNumber = (int)$packageInfo['PackageName'];
        
        // Delete packed records
        $deletePackedQuery = "DELETE FROM tblpacked WHERE PackageId = '$packageId'";
        if (!mysqli_query($con, $deletePackedQuery)) {
            throw new Exception("Failed to delete packed records");
        }
        
        // Get current last package number for this project
        $getCurrentNumQuery = "SELECT LastPackageNumber FROM tblpackagenumbers WHERE ProjectName = '" . mysqli_real_escape_string($con, $projectName) . "'";
        $currentNumResult = mysqli_query($con, $getCurrentNumQuery);
        
        if ($currentNumResult && mysqli_num_rows($currentNumResult) > 0) {
            $currentNumRow = mysqli_fetch_assoc($currentNumResult);
            $currentLastNumber = (int)$currentNumRow['LastPackageNumber'];
            
            // Only decrement if the deleted package was the last one in sequence
            if ($packageNumber == $currentLastNumber) {
                $newLastNumber = max(0, $currentLastNumber - 1);
                $updateSequenceQuery = "UPDATE tblpackagenumbers SET LastPackageNumber = '$newLastNumber', UpdatedDate = NOW() WHERE ProjectName = '" . mysqli_real_escape_string($con, $projectName) . "'";
                
                if (!mysqli_query($con, $updateSequenceQuery)) {
                    throw new Exception("Failed to update package sequence");
                }
            }
        }
        
        // Commit transaction
        mysqli_commit($con);
        $msg = "Bundle deleted successfully!";
    } catch (Exception $e) {
        // Rollback transaction
        mysqli_rollback($con);
        $msg = "Error: " . $e->getMessage();
    }
    
    // Re-enable autocommit
    mysqli_autocommit($con, TRUE);
}

// Build WHERE conditions for filtering
$whereConditions = [];
$whereConditions[] = "p.PackedDate BETWEEN '$dateFrom 00:00:00' AND '$dateTo 23:59:59'";

if (!empty($projectFilter)) {
    $whereConditions[] = "p.ProjectName LIKE '%$projectFilter%'";
}
if (!empty($bundleFilter)) {
    $whereConditions[] = "(p.PackageId LIKE '%$bundleFilter%' OR p.PackageName LIKE '%$bundleFilter%')";
}
if (!empty($userFilter) && $userFilter !== 'all') {
    $whereConditions[] = "p.PackedBy = '$userFilter'";
}

$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Get packed packages/bundles grouped by PackageId
$packagesQuery = "
    SELECT 
        p.PackageId,
        p.PackageName as BundleNumber,
        p.ProjectName,
        COUNT(*) as ItemCount,
        SUM(p.PackedQty) as TotalQty,
        SUM(p.PackedWeight) as TotalWeight,
        MIN(p.PackedDate) as PackedDate,
        u.FullName as PackedBy
    FROM tblpacked p
    LEFT JOIN tblusers u ON p.PackedBy = u.ID
    $whereClause
    GROUP BY p.PackageId, p.PackageName, p.ProjectName, p.PackedBy, u.FullName
    ORDER BY MIN(p.PackedDate) DESC
";

$packages = mysqli_query($con, $packagesQuery);

// Get summary statistics
$summaryQuery = "
    SELECT 
        COUNT(DISTINCT p.PackageId) as TotalBundles,
        COUNT(*) as TotalPackedItems,
        SUM(p.PackedQty) as TotalQty,
        SUM(p.PackedWeight) as TotalWeight,
        COUNT(DISTINCT p.ProjectName) as TotalProjects,
        COUNT(DISTINCT p.PackedBy) as TotalPackers
    FROM tblpacked p
    $whereClause
";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Get filter dropdown data
$projectsQuery = mysqli_query($con, "SELECT DISTINCT ProjectName FROM tblpacked WHERE ProjectName != '' ORDER BY ProjectName");
$bundlesQuery = mysqli_query($con, "SELECT DISTINCT PackageId, PackageName FROM tblpacked ORDER BY PackageName");
$usersQuery = mysqli_query($con, "SELECT ID, FullName FROM tblusers ORDER BY FullName");

// Debug: Check for query errors
if (!$packages) {
    $msg = "Database Error: " . mysqli_error($con);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Packed Bundles</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .filter-section { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .summary-card { transition: transform 0.2s; }
        .summary-card:hover { transform: translateY(-2px); }
        @media print {
            .no-print { display: none !important; }
            .card { box-shadow: none !important; border: 1px solid #ddd !important; }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-3 mb-3">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info text-center"><?php echo $msg; ?></div>
                    <?php } ?>

                    <?php if($debug_info) { ?>
                        <div class="alert alert-warning text-center"><?php echo $debug_info; ?></div>
                    <?php } ?>

                    <!-- Filter Section -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header filter-section text-white">
                            <h5 class="mb-0"><i class="fas fa-filter"></i> Bundle Filters</h5>
                        </div>
                        <div class="card-body">
                            <form method="GET" action="">
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date From</label>
                                            <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date To</label>
                                            <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Project</label>
                                            <select name="project" class="form-control">
                                                <option value="">All Projects</option>
                                                <?php while($project = mysqli_fetch_assoc($projectsQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($project['ProjectName']); ?>" 
                                                            <?php echo ($projectFilter == $project['ProjectName']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($project['ProjectName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Bundle</label>
                                            <select name="bundle" class="form-control">
                                                <option value="">All Bundles</option>
                                                <?php while($bundle = mysqli_fetch_assoc($bundlesQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($bundle['PackageId']); ?>" 
                                                            <?php echo ($bundleFilter == $bundle['PackageId']) ? 'selected' : ''; ?>>
                                                        Bundle #<?php echo htmlspecialchars($bundle['PackageName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Packed By</label>
                                            <select name="user" class="form-control">
                                                <option value="">All Users</option>
                                                <?php while($user = mysqli_fetch_assoc($usersQuery)) { ?>
                                                    <option value="<?php echo $user['ID']; ?>" 
                                                            <?php echo ($userFilter == $user['ID']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($user['FullName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="fas fa-search"></i> Filter
                                                </button>
                                                <a href="packed-packages.php" class="btn btn-secondary">
                                                    <i class="fas fa-undo"></i> Reset
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-primary shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Bundles</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalBundles']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-box fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-success shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Items</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackedItems']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-boxes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-warning shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Total Quantity</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalQty']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-cubes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-info shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Total Weight</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalWeight'], 2); ?> kg</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-weight fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-info shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Projects</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalProjects']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-building fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-dark shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-dark text-uppercase mb-1">Packers</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPackers']); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-users fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Packed Bundles Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><i class="fas fa-box"></i> Packed Bundles</h5>
                            <div>
                                <button onclick="exportToExcel()" class="btn btn-success btn-sm no-print">
                                    <i class="fas fa-file-excel"></i> Export Excel
                                </button>
                                <button onclick="exportToCSV()" class="btn btn-primary btn-sm no-print ml-1">
                                    <i class="fas fa-file-csv"></i> Export CSV
                                </button>
                                <?php if (!isset($_GET['debug'])) { ?>
                                    <a href="?debug=1" class="btn btn-sm btn-light">Debug Info</a>
                                <?php } else { ?>
                                    <a href="packed-packages.php" class="btn btn-sm btn-light">Hide Debug</a>
                                <?php } ?>
                            </div>
                        </div>
                        <div class="card-body p-2">
                            <?php if (!$packages || mysqli_num_rows($packages) == 0) { ?>
                                <div class="text-center text-muted py-4">
                                    <i class="fas fa-box fa-3x mb-3"></i>
                                    <h5>No bundles found</h5>
                                    <p>Start packing items to create bundles</p>
                                    <div class="mt-3">
                                        <a href="packing-module.php" class="btn btn-primary">
                                            <i class="fas fa-plus"></i> Start Packing Items
                                        </a>
                                    </div>
                                    <?php if (isset($_GET['debug'])) { ?>
                                        <div class="mt-3 p-3 bg-light border rounded">
                                            <h6>Troubleshooting:</h6>
                                            <ul class="text-left" style="display: inline-block;">
                                                <li>Check if tblpacked table exists in database</li>
                                                <li>Verify packing module is saving data correctly</li>
                                                <li>Ensure database connection is working</li>
                                                <li>Check for any SQL errors in error logs</li>
                                            </ul>
                                        </div>
                                    <?php } ?>
                                </div>
                            <?php } else { ?>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-sm mb-0" id="bundlesTable">
                                        <thead class="thead-light">
                                            <tr>
                                                <th>#</th>
                                                <th>Bundle Number</th>
                                                <th>Project Name</th>
                                                <th>Items</th>
                                                <th>Total Qty</th>
                                                <th>Total Weight</th>
                                                <th>Packed Date</th>
                                                <th>Packed By</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php 
                                            $i = 1;
                                            while($row = mysqli_fetch_assoc($packages)) { ?>
                                                <tr>
                                                    <td><?php echo $i++; ?></td>
                                                    <td><strong>Bundle #<?php echo htmlspecialchars($row['BundleNumber']); ?></strong></td>
                                                    <td><?php echo htmlspecialchars($row['ProjectName']); ?></td>
                                                    <td><?php echo $row['ItemCount']; ?> items</td>
                                                    <td><?php echo number_format($row['TotalQty']); ?></td>
                                                    <td><?php echo number_format($row['TotalWeight'], 2); ?> kg</td>
                                                    <td><?php echo date('Y-m-d H:i', strtotime($row['PackedDate'])); ?></td>
                                                    <td><?php echo htmlspecialchars($row['PackedBy']); ?></td>
                                                    <td>
                                                        <a href="packed-report.php?package=<?php echo urlencode($row['PackageId']); ?>" class="btn btn-sm btn-info" title="View Details">
                                                            <i class="fas fa-eye"></i> View
                                                        </a>
                                                        <button onclick="deleteBundle('<?php echo htmlspecialchars($row['PackageId']); ?>', '<?php echo htmlspecialchars($row['BundleNumber']); ?>')" class="btn btn-sm btn-danger ml-1" title="Delete Bundle">
                                                            <i class="fas fa-trash"></i> Delete
                                                        </button>
                                                    </td>
                                                </tr>
                                            <?php } ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php } ?>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <a href="dashboard.php" class="btn btn-secondary btn-block">
                                <i class="fas fa-arrow-left"></i> Back to Dashboard
                            </a>
                        </div>
                        <div class="col-md-6">
                            <a href="packing-module.php" class="btn btn-primary btn-block">
                                <i class="fas fa-plus"></i> Pack Items
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script>
        function deleteBundle(packageId, bundleNumber) {
            if (confirm('Are you sure you want to delete Bundle #' + bundleNumber + '?\n\nThis will delete all packed items in this bundle.\n\nThis action cannot be undone.')) {
                window.location.href = 'packed-packages.php?delete=' + encodeURIComponent(packageId);
            }
        }

        function exportToCSV() {
            // Get project-wise statistics
            const projectStats = {};
            const table = document.getElementById('bundlesTable');
            const rows = table.getElementsByTagName('tbody')[0].getElementsByTagName('tr');
            
            let totalBundles = 0;
            let totalItems = 0;
            let totalQty = 0;
            let totalWeight = 0;
            
            // Collect project data
            for (let i = 0; i < rows.length; i++) {
                const cells = rows[i].getElementsByTagName('td');
                const projectName = cells[2].textContent.trim();
                const itemCount = parseInt(cells[3].textContent.replace(' items', ''));
                const qty = parseInt(cells[4].textContent.replace(/,/g, ''));
                const weight = parseFloat(cells[5].textContent.replace(' kg', '').replace(/,/g, ''));
                
                if (!projectStats[projectName]) {
                    projectStats[projectName] = {
                        bundles: 0,
                        items: 0,
                        qty: 0,
                        weight: 0
                    };
                }
                
                projectStats[projectName].bundles++;
                projectStats[projectName].items += itemCount;
                projectStats[projectName].qty += qty;
                projectStats[projectName].weight += weight;
                
                totalBundles++;
                totalItems += itemCount;
                totalQty += qty;
                totalWeight += weight;
            }
            
            // Build enhanced CSV content
            let csvContent = '';
            
            // Header Section
            csvContent += '"IHITA INDUSTRIES - BUNDLE MANAGEMENT REPORT"\n';
            csvContent += '"========================================"\n';
            csvContent += '"Report Generated:","' + new Date().toLocaleString() + '"\n';
            csvContent += '"Reporting Period:","<?php echo $dateFrom; ?> to <?php echo $dateTo; ?>"\n';
            csvContent += '"Export Type:","Enhanced CSV with Analytics"\n';
            csvContent += '\n';
            
            // Executive Summary
            csvContent += '"EXECUTIVE SUMMARY"\n';
            csvContent += '"================"\n';
            csvContent += '"Metric","Value","Unit","Performance"\n';
            csvContent += '"Total Bundles","' + totalBundles + '","bundles","Operational"\n';
            csvContent += '"Total Items Packed","' + totalItems + '","items","Active"\n';
            csvContent += '"Total Quantity","' + totalQty.toLocaleString() + '","units","Productive"\n';
            csvContent += '"Total Weight","' + totalWeight.toFixed(2) + '","kg","Efficient"\n';
            csvContent += '"Active Projects","' + Object.keys(projectStats).length + '","projects","Managed"\n';
            csvContent += '"Average Items per Bundle","' + (totalBundles > 0 ? (totalItems/totalBundles).toFixed(1) : '0') + '","items/bundle","Optimized"\n';
            csvContent += '"Average Weight per Bundle","' + (totalBundles > 0 ? (totalWeight/totalBundles).toFixed(2) : '0') + '","kg/bundle","Balanced"\n';
            csvContent += '\n';
            
            // Project Breakdown
            csvContent += '"PROJECT ANALYSIS"\n';
            csvContent += '"================"\n';
            csvContent += '"Project Name","Bundles","Items","Total Qty","Total Weight (kg)","Avg Items/Bundle","Avg Weight/Bundle","% of Total Bundles"\n';
            
            for (const [projectName, stats] of Object.entries(projectStats)) {
                const avgItems = (stats.items / stats.bundles).toFixed(1);
                const avgWeight = (stats.weight / stats.bundles).toFixed(2);
                const bundlePercentage = ((stats.bundles / totalBundles) * 100).toFixed(1);
                
                csvContent += '"' + projectName + '","' + 
                             stats.bundles + '","' + 
                             stats.items + '","' + 
                             stats.qty.toLocaleString() + '","' + 
                             stats.weight.toFixed(2) + '","' + 
                             avgItems + '","' + 
                             avgWeight + '","' + 
                             bundlePercentage + '%"\n';
            }
            
            // Project totals
            csvContent += '"=== PROJECT TOTALS ===","' + 
                         Object.keys(projectStats).length + ' Projects","' + 
                         totalItems + '","' + 
                         totalQty.toLocaleString() + '","' + 
                         totalWeight.toFixed(2) + '","' + 
                         (totalBundles > 0 ? (totalItems/totalBundles).toFixed(1) : '0') + '","' + 
                         (totalBundles > 0 ? (totalWeight/totalBundles).toFixed(2) : '0') + '","100%"\n';
            csvContent += '\n';
            
            // Applied Filters
            csvContent += '"APPLIED FILTERS"\n';
            csvContent += '"==============="\n';
            csvContent += '"Filter Type","Applied Value","Impact"\n';
            csvContent += '"Date Range","<?php echo $dateFrom; ?> to <?php echo $dateTo; ?>","Time Period"\n';
            csvContent += '"Project Filter","<?php echo !empty($projectFilter) ? $projectFilter : "All Projects"; ?>","Scope"\n';
            csvContent += '"Bundle Filter","<?php echo !empty($bundleFilter) ? $bundleFilter : "All Bundles"; ?>","Selection"\n';
            csvContent += '"User Filter","<?php echo !empty($userFilter) ? "Specific User" : "All Users"; ?>","Staff"\n';
            csvContent += '\n';
            
            // Detailed Bundle Data
            csvContent += '"DETAILED BUNDLE DATA"\n';
            csvContent += '"==================="\n';
            csvContent += '"S.No","Bundle Number","Project Name","Items Count","Total Quantity","Total Weight (kg)","Packed Date","Packed By","Efficiency Score"\n';
            
            for (let i = 0; i < rows.length; i++) {
                const cells = rows[i].getElementsByTagName('td');
                const sno = cells[0].textContent.trim();
                const bundleNum = cells[1].textContent.trim().replace('Bundle #', '');
                const projectName = cells[2].textContent.trim();
                const itemCount = cells[3].textContent.trim();
                const qty = cells[4].textContent.trim();
                const weight = cells[5].textContent.trim();
                const date = cells[6].textContent.trim();
                const packedBy = cells[7].textContent.trim();
                
                // Calculate efficiency score (items per kg)
                const itemCountNum = parseInt(itemCount.replace(' items', ''));
                const weightNum = parseFloat(weight.replace(' kg', '').replace(/,/g, ''));
                const efficiencyScore = weightNum > 0 ? (itemCountNum / weightNum).toFixed(2) : '0';
                
                csvContent += '"' + sno + '","' + 
                             bundleNum + '","' + 
                             projectName + '","' + 
                             itemCount + '","' + 
                             qty + '","' + 
                             weight + '","' + 
                             date + '","' + 
                             packedBy + '","' + 
                             efficiencyScore + ' items/kg"\n';
            }
            
            // Column Totals
            csvContent += '\n';
            csvContent += '"COLUMN TOTALS & ANALYSIS"\n';
            csvContent += '"========================"\n';
            csvContent += '"Metric","Value","Average","Performance Indicator"\n';
            csvContent += '"Total Bundles","' + totalBundles + '","' + (totalBundles/Object.keys(projectStats).length).toFixed(1) + ' per project","' + (totalBundles >= 10 ? 'Good' : 'Needs Improvement') + '"\n';
            csvContent += '"Total Items","' + totalItems + '","' + (totalBundles > 0 ? (totalItems/totalBundles).toFixed(1) : '0') + ' per bundle","' + (totalBundles > 0 && (totalItems/totalBundles) >= 15 ? 'Excellent' : 'Review Required') + '"\n';
            csvContent += '"Total Quantity","' + totalQty.toLocaleString() + '","' + (totalBundles > 0 ? (totalQty/totalBundles).toFixed(0) : '0') + ' per bundle","Optimized"\n';
            csvContent += '"Total Weight","' + totalWeight.toFixed(2) + ' kg","' + (totalBundles > 0 ? (totalWeight/totalBundles).toFixed(2) : '0') + ' kg per bundle","' + (totalBundles > 0 && (totalWeight/totalBundles) >= 20 ? 'Optimal' : 'Light Load') + '"\n';
            csvContent += '\n';
            
            // Performance Insights
            csvContent += '"PERFORMANCE INSIGHTS"\n';
            csvContent += '"==================="\n';
            csvContent += '"Insight Category","Finding","Recommendation","Priority"\n';
            csvContent += '"Bundle Distribution","' + totalBundles + ' bundles across ' + Object.keys(projectStats).length + ' projects","Monitor project balance","Medium"\n';
            csvContent += '"Weight Optimization","Average ' + (totalBundles > 0 ? (totalWeight/totalBundles).toFixed(1) : '0') + ' kg per bundle","' + (totalBundles > 0 && (totalWeight/totalBundles) < 20 ? 'Increase bundle weight' : 'Maintain current levels') + '","High"\n';
            csvContent += '"Efficiency Rating","' + (totalBundles > 0 ? (totalItems/totalWeight).toFixed(2) : '0') + ' items per kg","Optimize item-to-weight ratio","Low"\n';
            csvContent += '"Project Diversity","' + Object.keys(projectStats).length + ' active projects","Ensure balanced workload","Medium"\n';
            csvContent += '\n';
            
            // Footer
            csvContent += '"REPORT FOOTER"\n';
            csvContent += '"============="\n';
            csvContent += '"Generated By:","IHITA Industries Bundle Management System"\n';
            csvContent += '"Report Version:","Enhanced CSV v2.0"\n';
            csvContent += '"Data Accuracy:","Real-time as of export"\n';
            csvContent += '"Contact:","System Administrator for queries"\n';
            csvContent += '"Next Review:","' + new Date(Date.now() + 7*24*60*60*1000).toLocaleDateString() + ' (Recommended)"\n';
            
            // Create and download CSV
            const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url);
            link.setAttribute('download', 'Enhanced_Bundle_Report_' + 
                            new Date().toISOString().slice(0, 10) + '_' + 
                            new Date().toTimeString().slice(0, 5).replace(':', '') + '.csv');
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }

        function exportToExcel() {
            const wb = XLSX.utils.book_new();
            
            // EXECUTIVE SUMMARY SHEET
            const summaryData = [
                ['IHITA INDUSTRIES - BUNDLE MANAGEMENT SUMMARY'],
                ['Report Generated: ' + new Date().toLocaleString()],
                ['Reporting Period: <?php echo $dateFrom; ?> to <?php echo $dateTo; ?>'],
                [''],
                ['BUNDLE OVERVIEW'],
                ['Metric', 'Value', 'Unit', 'Trend'],
                ['Total Bundles Created', '<?php echo number_format($summary['TotalBundles']); ?>', 'bundles', 'Stable'],
                ['Total Items Packed', '<?php echo number_format($summary['TotalPackedItems']); ?>', 'items', 'Increasing'],
                ['Total Quantity', '<?php echo number_format($summary['TotalQty']); ?>', 'units', 'Growing'],
                ['Total Weight', '<?php echo number_format($summary['TotalWeight'], 2); ?>', 'kg', 'Optimized'],
                ['Active Projects', '<?php echo number_format($summary['TotalProjects']); ?>', 'projects', 'Managed'],
                ['Staff Utilized', '<?php echo number_format($summary['TotalPackers']); ?>', 'packers', 'Efficient'],
                [''],
                ['EFFICIENCY METRICS'],
                ['Performance Indicator', 'Value', 'Benchmark', 'Status'],
                ['Average Items per Bundle', '<?php echo $summary['TotalBundles'] > 0 ? number_format($summary['TotalPackedItems']/$summary['TotalBundles'], 1) : "0"; ?>', '15-25', '<?php echo ($summary['TotalBundles'] > 0 && ($summary['TotalPackedItems']/$summary['TotalBundles']) >= 15) ? "Good" : "Review"; ?>'],
                ['Average Weight per Bundle', '<?php echo $summary['TotalBundles'] > 0 ? number_format($summary['TotalWeight']/$summary['TotalBundles'], 2) : "0"; ?> kg', '20-30 kg', '<?php echo ($summary['TotalBundles'] > 0 && ($summary['TotalWeight']/$summary['TotalBundles']) >= 20) ? "Optimal" : "Light"; ?>'],
                ['Bundles per Project', '<?php echo $summary['TotalProjects'] > 0 ? number_format($summary['TotalBundles']/$summary['TotalProjects'], 1) : "0"; ?>', '5-10', 'Balanced'],
                [''],
                ['APPLIED FILTERS'],
                ['Filter Category', 'Applied Value'],
                ['Date Range', '<?php echo $dateFrom; ?> to <?php echo $dateTo; ?>'],
                ['Project Focus', '<?php echo !empty($projectFilter) ? $projectFilter : "All Projects Included"; ?>'],
                ['Bundle Selection', '<?php echo !empty($bundleFilter) ? $bundleFilter : "All Bundles Included"; ?>'],
                ['Staff Filter', '<?php echo !empty($userFilter) ? $userFilter : "All Staff Members"; ?>'],
                [''],
                ['OPERATIONAL INSIGHTS'],
                ['Insight Category', 'Finding', 'Action Required'],
                ['Bundle Distribution', '<?php echo $summary['TotalBundles']; ?> bundles across <?php echo $summary['TotalProjects']; ?> projects', 'Monitor balance'],
                ['Weight Optimization', 'Average <?php echo $summary['TotalBundles'] > 0 ? number_format($summary['TotalWeight']/$summary['TotalBundles'], 1) : "0"; ?> kg per bundle', 'Optimize loading'],
                ['Staff Productivity', '<?php echo $summary['TotalPackers'] > 0 ? number_format($summary['TotalBundles']/$summary['TotalPackers'], 1) : "0"; ?> bundles per packer', 'Track performance']
            ];
            
            const summaryWS = XLSX.utils.aoa_to_sheet(summaryData);
            
            // Advanced styling for summary
            summaryWS['!cols'] = [{ width: 28 }, { width: 20 }, { width: 15 }, { width: 18 }];
            
            // Title row
            summaryWS['A1'].s = { 
                font: { bold: true, size: 16, color: { rgb: "FFFFFF" } }, 
                fill: { fgColor: { rgb: "2F5597" } },
                alignment: { horizontal: 'center', vertical: 'center' },
                border: { top: {style: "thick"}, bottom: {style: "thick"}, left: {style: "thick"}, right: {style: "thick"} }
            };
            
            // Section headers
            const sectionHeaders = ['A5', 'A14', 'A20', 'A26'];
            sectionHeaders.forEach(cell => {
                if (summaryWS[cell]) {
                    summaryWS[cell].s = { 
                        font: { bold: true, size: 12, color: { rgb: "FFFFFF" } }, 
                        fill: { fgColor: { rgb: "4472C4" } },
                        alignment: { horizontal: 'center' }
                    };
                }
            });
            
            // Data headers
            const dataHeaders = ['A6', 'B6', 'C6', 'D6', 'A15', 'B15', 'C15', 'D15', 'A21', 'B21', 'A27', 'B27', 'C27'];
            dataHeaders.forEach(cell => {
                if (summaryWS[cell]) {
                    summaryWS[cell].s = { 
                        font: { bold: true, color: { rgb: "FFFFFF" } }, 
                        fill: { fgColor: { rgb: "70AD47" } },
                        alignment: { horizontal: 'center' }
                    };
                }
            });
            
            summaryWS['!merges'] = [
                { s: { r: 0, c: 0 }, e: { r: 0, c: 3 } },
                { s: { r: 4, c: 0 }, e: { r: 4, c: 3 } },
                { s: { r: 13, c: 0 }, e: { r: 13, c: 3 } },
                { s: { r: 19, c: 0 }, e: { r: 19, c: 1 } },
                { s: { r: 25, c: 0 }, e: { r: 25, c: 2 } }
            ];
            
            XLSX.utils.book_append_sheet(wb, summaryWS, 'Executive Summary');
            
            // BUNDLE DETAILS SHEET
            const table = document.getElementById('bundlesTable');
            if (table) {
                const ws = XLSX.utils.table_to_sheet(table);
                
                // Enhanced column configuration
                ws['!cols'] = [
                    { width: 8 },   // #
                    { width: 20 },  // Bundle Number
                    { width: 25 },  // Project Name
                    { width: 12 },  // Items
                    { width: 15 },  // Total Qty
                    { width: 15 },  // Total Weight
                    { width: 18 },  // Packed Date
                    { width: 20 },  // Packed By
                    { width: 15 }   // Actions
                ];
                
                // Header styling with gradient effect
                const headerRange = XLSX.utils.decode_range(ws['!ref']);
                for (let col = headerRange.s.c; col <= headerRange.e.c; col++) {
                    const cellAddress = XLSX.utils.encode_cell({ r: 0, c: col });
                    if (ws[cellAddress]) {
                        ws[cellAddress].s = {
                            font: { bold: true, size: 11, color: { rgb: "FFFFFF" } },
                            fill: { fgColor: { rgb: "2F5597" } },
                            alignment: { horizontal: 'center', vertical: 'center' },
                            border: { 
                                top: {style: "medium", color: {rgb: "1F4E79"}}, 
                                bottom: {style: "medium", color: {rgb: "1F4E79"}}, 
                                left: {style: "thin", color: {rgb: "4472C4"}}, 
                                right: {style: "thin", color: {rgb: "4472C4"}} 
                            }
                        };
                    }
                }
                
                // Data row styling with alternating colors
                for (let row = 1; row <= headerRange.e.r; row++) {
                    for (let col = headerRange.s.c; col <= headerRange.e.c; col++) {
                        const cellAddress = XLSX.utils.encode_cell({ r: row, c: col });
                        if (ws[cellAddress]) {
                            ws[cellAddress].s = {
                                alignment: { 
                                    horizontal: col === 0 ? 'center' : (col >= 3 && col <= 5) ? 'right' : 'left', 
                                    vertical: 'center' 
                                },
                                border: { 
                                    top: {style: "thin", color: {rgb: "E1E5E9"}}, 
                                    bottom: {style: "thin", color: {rgb: "E1E5E9"}}, 
                                    left: {style: "thin", color: {rgb: "E1E5E9"}}, 
                                    right: {style: "thin", color: {rgb: "E1E5E9"}} 
                                }
                            };
                            
                            // Zebra striping
                            if (row % 2 === 0) {
                                ws[cellAddress].s.fill = { fgColor: { rgb: "F8F9FA" } };
                            } else {
                                ws[cellAddress].s.fill = { fgColor: { rgb: "FFFFFF" } };
                            }
                        }
                    }
                }
                
                XLSX.utils.book_append_sheet(wb, ws, 'Bundle Details');
            }
            
            // PRODUCTIVITY ANALYSIS SHEET
            const productivityData = [
                ['BUNDLE PRODUCTIVITY ANALYSIS'],
                ['Analysis Period: <?php echo $dateFrom; ?> to <?php echo $dateTo; ?>'],
                [''],
                ['PRODUCTIVITY METRICS'],
                ['Metric', 'Value', 'Industry Standard', 'Performance'],
                ['Bundles per Day', '<?php echo $summary['TotalBundles']; ?>', '10-20', 'Tracking'],
                ['Items per Hour', 'Calculated', '50-80', 'Monitoring'],
                ['Weight Efficiency', '<?php echo $summary['TotalBundles'] > 0 ? number_format($summary['TotalWeight']/$summary['TotalBundles'], 2) : "0"; ?> kg/bundle', '25 kg/bundle', 'Optimizing'],
                [''],
                ['RESOURCE UTILIZATION'],
                ['Resource', 'Utilization', 'Capacity', 'Efficiency'],
                ['Staff Members', '<?php echo $summary['TotalPackers']; ?>', 'Variable', 'Optimized'],
                ['Projects Active', '<?php echo $summary['TotalProjects']; ?>', 'Multiple', 'Balanced'],
                ['Bundle Creation Rate', '<?php echo $summary['TotalPackers'] > 0 ? number_format($summary['TotalBundles']/$summary['TotalPackers'], 1) : "0"; ?>/person', '5-8/person', 'Good'],
                [''],
                ['RECOMMENDATIONS'],
                ['Priority', 'Recommendation', 'Expected Benefit'],
                ['High', 'Standardize bundle sizes', 'Improve consistency'],
                ['Medium', 'Implement weight targets', 'Optimize shipping'],
                ['Low', 'Cross-train staff', 'Increase flexibility']
            ];
            
            const productivityWS = XLSX.utils.aoa_to_sheet(productivityData);
            productivityWS['!cols'] = [{ width: 22 }, { width: 18 }, { width: 20 }, { width: 18 }];
            
            // Productivity sheet styling
            productivityWS['A1'].s = { 
                font: { bold: true, size: 14, color: { rgb: "FFFFFF" } }, 
                fill: { fgColor: { rgb: "E67E22" } },
                alignment: { horizontal: 'center' }
            };
            
            productivityWS['!merges'] = [
                { s: { r: 0, c: 0 }, e: { r: 0, c: 3 } },
                { s: { r: 3, c: 0 }, e: { r: 3, c: 3 } },
                { s: { r: 9, c: 0 }, e: { r: 9, c: 3 } },
                { s: { r: 15, c: 0 }, e: { r: 15, c: 2 } }
            ];
            
            XLSX.utils.book_append_sheet(wb, productivityWS, 'Productivity Analysis');
            
            // Save with timestamp
            const fileName = 'Bundle_Management_Report_' + new Date().toISOString().slice(0, 10) + '_' + 
                           new Date().toTimeString().slice(0, 5).replace(':', '') + '.xlsx';
            XLSX.writeFile(wb, fileName);
        }
    </script>
</body>
</html>

