<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];
$msg = "";

// Initialize filter variables
$dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01', strtotime('-3 months'));
$dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
$projectFilter = isset($_GET['project']) ? mysqli_real_escape_string($con, $_GET['project']) : '';
$roomFilter = isset($_GET['room']) ? mysqli_real_escape_string($con, $_GET['room']) : '';
$productFilter = isset($_GET['product']) ? mysqli_real_escape_string($con, $_GET['product']) : '';
$userFilter = isset($_GET['user']) ? mysqli_real_escape_string($con, $_GET['user']) : '';
$statusFilter = isset($_GET['status']) ? $_GET['status'] : 'all';

// Build WHERE conditions
$whereConditions = [];
$whereConditions[] = "i.UploadDate BETWEEN '$dateFrom 00:00:00' AND '$dateTo 23:59:59'";

if (!empty($projectFilter)) {
    $whereConditions[] = "i.ProjectName LIKE '%$projectFilter%'";
}
if (!empty($roomFilter)) {
    $whereConditions[] = "i.RoomNo LIKE '%$roomFilter%'";
}
if (!empty($productFilter)) {
    $whereConditions[] = "i.ProductName LIKE '%$productFilter%'";
}
if (!empty($userFilter) && $userFilter !== 'all') {
    $whereConditions[] = "i.UploadedBy = '$userFilter'";
}

$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Main query to get inventory with packed quantities
$reportQuery = "
    SELECT 
        i.ID,
        i.Barcode,
        i.ProductName,
        i.Qty as StockIn,
        i.Weight as WeightIn,
        i.RoomNo,
        i.ProjectName,
        i.UploadId,
        i.UploadDate,
        u.FullName as UploadedBy,
        COALESCE(SUM(p.PackedQty), 0) as PackedQty,
        COALESCE(SUM(p.PackedWeight), 0) as PackedWeight,
        (i.Qty - COALESCE(SUM(p.PackedQty), 0)) as RemainingQty,
        (i.Weight - COALESCE(SUM(p.PackedWeight), 0)) as RemainingWeight,
        CASE 
            WHEN (i.Qty - COALESCE(SUM(p.PackedQty), 0)) <= 0 THEN 'Fully Packed'
            WHEN COALESCE(SUM(p.PackedQty), 0) > 0 THEN 'Partially Packed'
            ELSE 'Not Packed'
        END as Status
    FROM tblinventory i
    LEFT JOIN tblusers u ON i.UploadedBy = u.ID
    LEFT JOIN tblpacked p ON i.ID = p.ProductId
    $whereClause
    GROUP BY i.ID, i.Barcode, i.ProductName, i.Qty, i.Weight, i.RoomNo, i.ProjectName, i.UploadId, i.UploadDate, u.FullName
";

// Apply status filter after grouping
if ($statusFilter !== 'all') {
    $reportQuery = "SELECT * FROM ($reportQuery) as filtered_data WHERE ";
    switch($statusFilter) {
        case 'not_packed':
            $reportQuery .= "Status = 'Not Packed'";
            break;
        case 'partially_packed':
            $reportQuery .= "Status = 'Partially Packed'";
            break;
        case 'fully_packed':
            $reportQuery .= "Status = 'Fully Packed'";
            break;
    }
}

$reportQuery .= " ORDER BY UploadDate DESC, ProductName ASC";

$reportResult = mysqli_query($con, $reportQuery);

// For summary statistics, we need the base query without status filtering
$baseSummaryQuery = "
    SELECT 
        i.ID,
        i.Barcode,
        i.ProductName,
        i.Qty as StockIn,
        i.Weight as WeightIn,
        i.RoomNo,
        i.ProjectName,
        i.UploadId,
        i.UploadDate,
        u.FullName as UploadedBy,
        COALESCE(SUM(p.PackedQty), 0) as PackedQty,
        COALESCE(SUM(p.PackedWeight), 0) as PackedWeight,
        (i.Qty - COALESCE(SUM(p.PackedQty), 0)) as RemainingQty,
        (i.Weight - COALESCE(SUM(p.PackedWeight), 0)) as RemainingWeight,
        CASE 
            WHEN (i.Qty - COALESCE(SUM(p.PackedQty), 0)) <= 0 THEN 'Fully Packed'
            WHEN COALESCE(SUM(p.PackedQty), 0) > 0 THEN 'Partially Packed'
            ELSE 'Not Packed'
        END as Status
    FROM tblinventory i
    LEFT JOIN tblusers u ON i.UploadedBy = u.ID
    LEFT JOIN tblpacked p ON i.ID = p.ProductId
    $whereClause
    GROUP BY i.ID, i.Barcode, i.ProductName, i.Qty, i.Weight, i.RoomNo, i.ProjectName, i.UploadId, i.UploadDate, u.FullName
";

// Get summary statistics
$summaryQuery = "
    SELECT 
        COUNT(*) as TotalProducts,
        COALESCE(SUM(StockIn), 0) as TotalStockIn,
        COALESCE(SUM(WeightIn), 0) as TotalWeightIn,
        COALESCE(SUM(PackedQty), 0) as TotalPacked,
        COALESCE(SUM(PackedWeight), 0) as TotalPackedWeight,
        COALESCE(SUM(RemainingQty), 0) as TotalRemaining,
        COALESCE(SUM(RemainingWeight), 0) as TotalRemainingWeight,
        SUM(CASE WHEN Status = 'Not Packed' THEN 1 ELSE 0 END) as NotPackedCount,
        SUM(CASE WHEN Status = 'Partially Packed' THEN 1 ELSE 0 END) as PartiallyPackedCount,
        SUM(CASE WHEN Status = 'Fully Packed' THEN 1 ELSE 0 END) as FullyPackedCount
    FROM ($baseSummaryQuery) as summary_data
";
$summaryResult = mysqli_query($con, $summaryQuery);

// Add error checking
if (!$reportResult) {
    die('Error in main query: ' . mysqli_error($con));
}
if (!$summaryResult) {
    die('Error in summary query: ' . mysqli_error($con));
}

$summary = mysqli_fetch_assoc($summaryResult);

// Debug mode - shows query details
$debugMode = isset($_GET['debug']) && $_GET['debug'] == '1';
if ($debugMode) {
    echo "<div class='alert alert-info'>";
    echo "<h5>Debug Information:</h5>";
    echo "<p><strong>Date Range:</strong> $dateFrom to $dateTo</p>";
    echo "<p><strong>Main Query Row Count:</strong> " . mysqli_num_rows($reportResult) . "</p>";
    echo "<p><strong>Summary Totals:</strong></p>";
    echo "<ul>";
    echo "<li>Total Products: " . ($summary['TotalProducts'] ?? 0) . "</li>";
    echo "<li>Total Stock In: " . ($summary['TotalStockIn'] ?? 0) . "</li>";
    echo "<li>Total Packed: " . ($summary['TotalPacked'] ?? 0) . "</li>";
    echo "<li>Total Remaining: " . ($summary['TotalRemaining'] ?? 0) . "</li>";
    echo "<li>Not Packed Count: " . ($summary['NotPackedCount'] ?? 0) . "</li>";
    echo "<li>Partially Packed Count: " . ($summary['PartiallyPackedCount'] ?? 0) . "</li>";
    echo "<li>Fully Packed Count: " . ($summary['FullyPackedCount'] ?? 0) . "</li>";
    echo "</ul>";
    
    // Show detailed breakdown of items with remaining quantities
    echo "<h6>Items with Remaining Quantities:</h6>";
    $debugQuery = "
        SELECT 
            i.Barcode,
            i.ProductName,
            i.Qty as StockIn,
            COALESCE(SUM(p.PackedQty), 0) as PackedQty,
            (i.Qty - COALESCE(SUM(p.PackedQty), 0)) as RemainingQty,
            CASE 
                WHEN (i.Qty - COALESCE(SUM(p.PackedQty), 0)) = 0 THEN 'Fully Packed'
                WHEN COALESCE(SUM(p.PackedQty), 0) > 0 THEN 'Partially Packed'
                ELSE 'Not Packed'
            END as Status
        FROM tblinventory i
        LEFT JOIN tblpacked p ON i.ID = p.ProductId
        WHERE i.UploadDate BETWEEN '$dateFrom 00:00:00' AND '$dateTo 23:59:59'
        GROUP BY i.ID, i.Barcode, i.ProductName, i.Qty
        HAVING RemainingQty > 0
        ORDER BY RemainingQty DESC
        LIMIT 10
    ";
    $debugResult = mysqli_query($con, $debugQuery);
    if (mysqli_num_rows($debugResult) > 0) {
        echo "<table class='table table-sm'>";
        echo "<tr><th>Barcode</th><th>Product</th><th>Stock In</th><th>Packed</th><th>Remaining</th><th>Status</th></tr>";
        while ($debugRow = mysqli_fetch_assoc($debugResult)) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($debugRow['Barcode']) . "</td>";
            echo "<td>" . htmlspecialchars($debugRow['ProductName']) . "</td>";
            echo "<td>" . $debugRow['StockIn'] . "</td>";
            echo "<td>" . $debugRow['PackedQty'] . "</td>";
            echo "<td><strong>" . $debugRow['RemainingQty'] . "</strong></td>";
            echo "<td>" . $debugRow['Status'] . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p class='text-muted'>No items with remaining quantities found in the selected date range.</p>";
    }
    
    echo "</div>";
}

// Get filter dropdown data
$projectsQuery = mysqli_query($con, "SELECT DISTINCT ProjectName FROM tblinventory WHERE ProjectName != '' ORDER BY ProjectName");
$roomsQuery = mysqli_query($con, "SELECT DISTINCT RoomNo FROM tblinventory WHERE RoomNo != '' ORDER BY RoomNo");
$usersQuery = mysqli_query($con, "SELECT ID, FullName FROM tblusers ORDER BY FullName");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Inventory Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .status-not-packed { 
            background-color: #f8d7da; 
            color: #721c24; 
            font-weight: bold;
        }
        .status-partially-packed { 
            background-color: #fff3cd; 
            color: #856404; 
            font-weight: bold;
            border-left: 4px solid #ffc107;
        }
        .status-fully-packed { 
            background-color: #d4edda; 
            color: #155724; 
        }
        .pending-highlight {
            background: linear-gradient(90deg, #fff3cd 0%, #ffffff 100%);
            border-left: 3px solid #fd7e14;
        }
        .remaining-qty {
            font-size: 1.1em;
            font-weight: bold;
            color: #fd7e14;
        }
        .filter-section { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .summary-card { transition: transform 0.2s; }
        .summary-card:hover { transform: translateY(-2px); }
        @media print {
            .no-print { display: none !important; }
            .card { box-shadow: none !important; border: 1px solid #ddd !important; }
        }
        @media (max-width: 768px) {
            .table-responsive table, .table-responsive thead, .table-responsive tbody, 
            .table-responsive th, .table-responsive td, .table-responsive tr {
                display: block;
            }
            .table-responsive thead tr { position: absolute; top: -9999px; left: -9999px; }
            .table-responsive tr { border: 1px solid #ccc; margin-bottom: 10px; }
            .table-responsive td {
                border: none; position: relative; padding-left: 50% !important;
            }
            .table-responsive td:before {
                content: attr(data-label) ": "; position: absolute; left: 6px; width: 45%;
                padding-right: 10px; white-space: nowrap; font-weight: bold;
            }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-3 mb-3">
                    
                    <!-- Quick Access for Pending Items -->
                    <?php if (($summary['NotPackedCount'] ?? 0) > 0 || ($summary['PartiallyPackedCount'] ?? 0) > 0) { ?>
                    <div class="alert alert-warning alert-dismissible fade show" role="alert">
                        <h5><i class="fas fa-exclamation-triangle"></i> Items Requiring Attention</h5>
                        <div class="row">
                            <?php if (($summary['NotPackedCount'] ?? 0) > 0) { ?>
                            <div class="col-md-6">
                                <strong><?php echo number_format($summary['NotPackedCount']); ?> items not packed yet</strong><br>
                                <a href="?status=not_packed&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" class="btn btn-danger btn-sm">
                                    <i class="fas fa-eye"></i> View Not Packed Items
                                </a>
                            </div>
                            <?php } ?>
                            <?php if (($summary['PartiallyPackedCount'] ?? 0) > 0) { ?>
                            <div class="col-md-6">
                                <strong><?php echo number_format($summary['PartiallyPackedCount']); ?> items partially packed</strong><br>
                                <a href="?status=partially_packed&date_from=<?php echo $dateFrom; ?>&date_to=<?php echo $dateTo; ?>" class="btn btn-warning btn-sm">
                                    <i class="fas fa-eye"></i> View Partially Packed Items
                                </a>
                            </div>
                            <?php } ?>
                        </div>
                        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <?php } ?>
                    
                    <!-- Filter Section -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header filter-section text-white">
                            <h5 class="mb-0"><i class="fas fa-filter"></i> Inventory Report Filters</h5>
                        </div>
                        <div class="card-body">
                            <form method="GET" action="">
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date From</label>
                                            <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Date To</label>
                                            <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Project</label>
                                            <select name="project" class="form-control">
                                                <option value="">All Projects</option>
                                                <?php while($project = mysqli_fetch_assoc($projectsQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($project['ProjectName']); ?>" 
                                                            <?php echo ($projectFilter == $project['ProjectName']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($project['ProjectName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Room</label>
                                            <select name="room" class="form-control">
                                                <option value="">All Rooms</option>
                                                <?php while($room = mysqli_fetch_assoc($roomsQuery)) { ?>
                                                    <option value="<?php echo htmlspecialchars($room['RoomNo']); ?>" 
                                                            <?php echo ($roomFilter == $room['RoomNo']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($room['RoomNo']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Product Name</label>
                                            <input type="text" name="product" class="form-control" placeholder="Search product..." value="<?php echo htmlspecialchars($productFilter); ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Uploaded By</label>
                                            <select name="user" class="form-control">
                                                <option value="">All Users</option>
                                                <?php while($user = mysqli_fetch_assoc($usersQuery)) { ?>
                                                    <option value="<?php echo $user['ID']; ?>" 
                                                            <?php echo ($userFilter == $user['ID']) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($user['FullName']); ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Status</label>
                                            <select name="status" class="form-control">
                                                <option value="all" <?php echo ($statusFilter == 'all') ? 'selected' : ''; ?>>All Status</option>
                                                <option value="not_packed" <?php echo ($statusFilter == 'not_packed') ? 'selected' : ''; ?>>Not Packed</option>
                                                <option value="partially_packed" <?php echo ($statusFilter == 'partially_packed') ? 'selected' : ''; ?>>Partially Packed</option>
                                                <option value="fully_packed" <?php echo ($statusFilter == 'fully_packed') ? 'selected' : ''; ?>>Fully Packed</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary">
                                                    <i class="fas fa-search"></i> Filter
                                                </button>
                                                <a href="inventory-report.php" class="btn btn-secondary">
                                                    <i class="fas fa-undo"></i> Reset
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-primary shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Products</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalProducts'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-boxes fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-success shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Stock In</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalStockIn'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-down fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-warning shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Packed Out</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalPacked'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-up fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-info shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Remaining</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['TotalRemaining'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-balance-scale fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-danger shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Not Packed</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['NotPackedCount'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
                            <div class="card border-left-success shadow h-100 py-2 summary-card">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Fully Packed</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($summary['FullyPackedCount'] ?? 0); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header bg-primary text-white">
                            <div class="row align-items-center">
                                <div class="col">
                                    <h5 class="mb-0"><i class="fas fa-chart-bar"></i> Inventory Stock Report</h5>
                                </div>
                                <div class="col-auto no-print">
                                    <button onclick="window.print()" class="btn btn-light btn-sm">
                                        <i class="fas fa-print"></i> Print Report
                                    </button>
                                    <button onclick="exportToCSV()" class="btn btn-success btn-sm">
                                        <i class="fas fa-file-csv"></i> Export CSV
                                    </button>
                                    <button onclick="exportToExcel()" class="btn btn-info btn-sm">
                                        <i class="fas fa-file-excel"></i> Export Excel
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="card-body p-2">
                            <?php if (mysqli_num_rows($reportResult) == 0) { ?>
                                <div class="text-center text-muted py-4">
                                    <i class="fas fa-search fa-3x mb-3"></i>
                                    <h5>No data found</h5>
                                    <p>Try adjusting your filter criteria</p>
                                </div>
                            <?php } else { ?>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-sm mb-0" id="reportTable">
                                        <thead class="thead-light">
                                            <tr>
                                                <th>#</th>
                                                <th>Barcode</th>
                                                <th>Product Name</th>
                                                <th>Project</th>
                                                <th>Room</th>
                                                <th>Stock In</th>
                                                <th>Weight In</th>
                                                <th>Packed Out</th>
                                                <th>Weight Out</th>
                                                <th>Remaining</th>
                                                <th>Remaining Weight</th>
                                                <th>Status</th>
                                                <th>Upload Date</th>
                                                <th>Uploaded By</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php $i=1; while($row = mysqli_fetch_assoc($reportResult)) { 
                                                $hasPendingQty = ($row['RemainingQty'] ?? 0) > 0;
                                                $statusClass = $row['Status'] == 'Not Packed' ? 'status-not-packed' : 
                                                    ($row['Status'] == 'Partially Packed' ? 'status-partially-packed' : 'status-fully-packed');
                                                if ($hasPendingQty && $row['Status'] == 'Partially Packed') {
                                                    $statusClass .= ' pending-highlight';
                                                }
                                            ?>
                                                <tr class="<?php echo $statusClass; ?>">
                                                    <td data-label="#"><?php echo $i++; ?></td>
                                                    <td data-label="Barcode"><?php echo htmlspecialchars($row['Barcode']); ?></td>
                                                    <td data-label="Product Name"><?php echo htmlspecialchars($row['ProductName']); ?></td>
                                                    <td data-label="Project"><?php echo htmlspecialchars($row['ProjectName']); ?></td>
                                                    <td data-label="Room"><?php echo htmlspecialchars($row['RoomNo']); ?></td>
                                                    <td data-label="Stock In"><strong><?php echo number_format($row['StockIn'] ?? 0); ?></strong></td>
                                                    <td data-label="Weight In"><strong><?php echo number_format($row['WeightIn'] ?? 0, 2); ?> kg</strong></td>
                                                    <td data-label="Packed Out"><?php echo number_format($row['PackedQty'] ?? 0); ?></td>
                                                    <td data-label="Weight Out"><?php echo number_format($row['PackedWeight'] ?? 0, 2); ?> kg</td>
                                                    <td data-label="Remaining">
                                                        <span class="<?php echo $hasPendingQty ? 'remaining-qty' : 'text-muted'; ?>">
                                                            <?php echo number_format($row['RemainingQty'] ?? 0); ?>
                                                            <?php if ($hasPendingQty) { ?>
                                                                <i class="fas fa-exclamation-triangle ml-1" title="Pending to pack"></i>
                                                            <?php } ?>
                                                        </span>
                                                    </td>
                                                    <td data-label="Remaining Weight">
                                                        <span class="<?php echo $hasPendingQty ? 'remaining-qty' : 'text-muted'; ?>">
                                                            <?php echo number_format($row['RemainingWeight'] ?? 0, 2); ?> kg
                                                            <?php if ($hasPendingQty) { ?>
                                                                <i class="fas fa-exclamation-triangle ml-1" title="Pending to pack"></i>
                                                            <?php } ?>
                                                        </span>
                                                    </td>
                                                    <td data-label="Status">
                                                        <span class="badge badge-<?php 
                                                            echo $row['Status'] == 'Not Packed' ? 'danger' : 
                                                                ($row['Status'] == 'Partially Packed' ? 'warning' : 'success'); 
                                                        ?>"><?php echo $row['Status']; ?></span>
                                                        <?php if ($hasPendingQty) { ?>
                                                            <br><small class="text-muted"><?php echo number_format($row['RemainingQty']); ?> pending</small>
                                                        <?php } ?>
                                                    </td>
                                                    <td data-label="Upload Date"><?php echo date('Y-m-d', strtotime($row['UploadDate'])); ?></td>
                                                    <td data-label="Uploaded By"><?php echo htmlspecialchars($row['UploadedBy']); ?></td>
                                                </tr>
                                            <?php } ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php } ?>
                        </div>
                    </div>

                    <div class="row no-print">
                        <div class="col-md-6">
                            <a href="dashboard.php" class="btn btn-secondary btn-block">
                                <i class="fas fa-arrow-left"></i> Back to Dashboard
                            </a>
                        </div>
                        <div class="col-md-6">
                            <a href="inventory-management.php" class="btn btn-primary btn-block">
                                <i class="fas fa-upload"></i> Manage Inventory
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    
    <script>
        function exportToCSV() {
            const table = document.getElementById('reportTable');
            const rows = Array.from(table.querySelectorAll('tr'));
            let csvContent = '';

            // --- Project/Filter Details Header ---
            csvContent += 'Inventory Stock Report\n';
            csvContent += 'Generated:,' + new Date().toLocaleString() + '\n';
            csvContent += 'Period:,<?php echo $dateFrom; ?> to <?php echo $dateTo; ?>\n';
            csvContent += 'Project:,<?php echo $projectFilter ? $projectFilter : "All Projects"; ?>\n';
            csvContent += 'Room:,<?php echo $roomFilter ? $roomFilter : "All Rooms"; ?>\n';
            csvContent += 'Product:,<?php echo $productFilter ? $productFilter : "All Products"; ?>\n';
            csvContent += 'Uploaded By:,<?php echo $userFilter ? $userFilter : "All Users"; ?>\n';
            csvContent += 'Status:,<?php echo ucfirst(str_replace("_", " ", $statusFilter)); ?>\n';
            csvContent += '\n';

            // --- Table Header and Data ---
            let dataRows = [];
            let stockInSum = 0, packedOutSum = 0, remainingSum = 0, weightInSum = 0, packedWeightSum = 0, remainingWeightSum = 0;
            rows.forEach((row, idx) => {
                const cols = Array.from(row.querySelectorAll('td, th'));
                const rowData = cols.map(col => {
                    let text = col.textContent.trim();
                    text = text.replace(/\s+/g, ' ');
                    if (text.includes(',') || text.includes('"')) {
                        text = '"' + text.replace(/"/g, '""') + '"';
                    }
                    return text;
                });
                if (idx === 0) {
                    dataRows.push(rowData); // header
                } else if (cols.length > 0) {
                    // Sum columns for summary (Stock In: 5, Weight In: 6, Packed Out: 7, Packed Weight: 8, Remaining: 9, Remaining Weight: 10)
                    stockInSum += parseFloat(cols[5].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    weightInSum += parseFloat(cols[6].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    packedOutSum += parseFloat(cols[7].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    packedWeightSum += parseFloat(cols[8].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    remainingSum += parseFloat(cols[9].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    remainingWeightSum += parseFloat(cols[10].textContent.replace(/[^0-9.-]/g, '')) || 0;
                    dataRows.push(rowData);
                }
            });

            // Write table rows
            dataRows.forEach(rowArr => {
                csvContent += rowArr.join(',') + '\n';
            });

            // --- Summary Row ---
            csvContent += '\n';
            csvContent += 'SUMMARY\n';
            csvContent += 'Total Products:,<?php echo number_format($summary['TotalProducts'] ?? 0); ?>\n';
            csvContent += 'Total Stock In:,' + stockInSum + '\n';
            csvContent += 'Total Weight In:,' + weightInSum.toFixed(2) + ' kg\n';
            csvContent += 'Total Packed Out:,' + packedOutSum + '\n';
            csvContent += 'Total Packed Weight:,' + packedWeightSum.toFixed(2) + ' kg\n';
            csvContent += 'Total Remaining:,' + remainingSum + '\n';
            csvContent += 'Total Remaining Weight:,' + remainingWeightSum.toFixed(2) + ' kg\n';
            csvContent += 'Not Packed: ,<?php echo number_format($summary['NotPackedCount'] ?? 0); ?>\n';
            csvContent += 'Fully Packed: ,<?php echo number_format($summary['FullyPackedCount'] ?? 0); ?>\n';

            // Download
            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'inventory_report_<?php echo date('Y-m-d'); ?>.csv';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        }

        function exportToExcel() {
            const wb = XLSX.utils.book_new();
            
            // EXECUTIVE SUMMARY SHEET
            const summaryData = [
                ['IHITA INDUSTRIES - INVENTORY MANAGEMENT DASHBOARD'],
                ['Report Generated: ' + new Date().toLocaleString()],
                ['Analysis Period: <?php echo $dateFrom; ?> to <?php echo $dateTo; ?>'],
                [''],
                ['INVENTORY OVERVIEW'],
                ['Key Performance Indicator', 'Current Value', 'Target', 'Status'],
                ['Total Products Managed', '<?php echo number_format($summary['TotalProducts'] ?? 0); ?>', 'Increasing', 'Tracked'],
                ['Stock In (Quantity)', '<?php echo number_format($summary['TotalStockIn'] ?? 0); ?>', 'Optimized', 'Managed'],
                ['Stock In (Weight)', '<?php echo number_format($summary['TotalWeightIn'] ?? 0, 2); ?> kg', 'Balanced', 'Controlled'],
                ['Packed Out (Quantity)', '<?php echo number_format($summary['TotalPacked'] ?? 0); ?>', 'Efficient', 'Active'],
                ['Packed Out (Weight)', '<?php echo number_format($summary['TotalPackedWeight'] ?? 0, 2); ?> kg', 'Optimized', 'Flowing'],
                ['Remaining Stock (Quantity)', '<?php echo number_format($summary['TotalRemaining'] ?? 0); ?>', 'Available', 'Ready'],
                ['Remaining Stock (Weight)', '<?php echo number_format($summary['TotalRemainingWeight'] ?? 0, 2); ?> kg', 'Sufficient', 'Adequate'],
                [''],
                ['OPERATIONAL STATUS BREAKDOWN'],
                ['Status Category', 'Item Count', 'Percentage', 'Priority Level', 'Action Required'],
                ['Not Packed', '<?php echo number_format($summary['NotPackedCount'] ?? 0); ?>', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format((($summary['NotPackedCount'] ?? 0)/($summary['TotalProducts'] ?? 1))*100, 1) : "0"; ?>%', 'High', 'Schedule for packing'],
                ['Partially Packed', '<?php echo number_format($summary['PartiallyPackedCount'] ?? 0); ?>', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format((($summary['PartiallyPackedCount'] ?? 0)/($summary['TotalProducts'] ?? 1))*100, 1) : "0"; ?>%', 'Medium', 'Complete packing'],
                ['Fully Packed', '<?php echo number_format($summary['FullyPackedCount'] ?? 0); ?>', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format((($summary['FullyPackedCount'] ?? 0)/($summary['TotalProducts'] ?? 1))*100, 1) : "0"; ?>%', 'Low', 'Monitor for restock'],
                [''],
                ['EFFICIENCY METRICS'],
                ['Performance Metric', 'Current Rate', 'Industry Benchmark', 'Performance Rating'],
                ['Packing Efficiency Rate', '<?php echo $summary['TotalStockIn'] > 0 ? number_format(($summary['TotalPacked']/$summary['TotalStockIn'])*100, 2) : "0"; ?>%', '85-95%', '<?php echo ($summary['TotalStockIn'] > 0 && ($summary['TotalPacked']/$summary['TotalStockIn']) >= 0.85) ? "Excellent" : "Improving"; ?>'],
                ['Inventory Turnover', '<?php echo $summary['TotalStockIn'] > 0 ? number_format(($summary['TotalPacked']/$summary['TotalStockIn'])*100, 2) : "0"; ?>%', '80-90%', 'Good'],
                ['Weight Utilization', '<?php echo $summary['TotalWeightIn'] > 0 ? number_format(($summary['TotalPackedWeight']/$summary['TotalWeightIn'])*100, 2) : "0"; ?>%', '75-85%', 'Optimal'],
                ['Stock Availability', '<?php echo $summary['TotalStockIn'] > 0 ? number_format(($summary['TotalRemaining']/$summary['TotalStockIn'])*100, 2) : "0"; ?>%', '15-25%', 'Balanced'],
                [''],
                ['FILTER CONFIGURATION'],
                ['Filter Type', 'Applied Setting'],
                ['Date Range', '<?php echo $dateFrom; ?> to <?php echo $dateTo; ?>'],
                ['Project Scope', '<?php echo !empty($projectFilter) ? $projectFilter : "All Projects"; ?>'],
                ['Room Coverage', '<?php echo !empty($roomFilter) ? $roomFilter : "All Rooms"; ?>'],
                ['Product Selection', '<?php echo !empty($productFilter) ? $productFilter : "All Products"; ?>'],
                ['User Focus', '<?php echo !empty($userFilter) ? $userFilter : "All Users"; ?>'],
                ['Status Filter', '<?php echo ucfirst(str_replace("_", " ", $statusFilter)); ?>'],
                [''],
                ['STRATEGIC INSIGHTS'],
                ['Insight Area', 'Current State', 'Recommendation'],
                ['Inventory Balance', '<?php echo number_format($summary['TotalRemaining'] ?? 0); ?> items remaining', 'Monitor reorder points'],
                ['Packing Flow', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format((($summary['TotalPacked'] ?? 0)/($summary['TotalProducts'] ?? 1))*100, 1) : "0"; ?>% completion rate', 'Optimize workflow'],
                ['Resource Allocation', '<?php echo $summary['NotPackedCount'] ?? 0; ?> items pending', 'Prioritize high-value items']
            ];
            
            const summaryWS = XLSX.utils.aoa_to_sheet(summaryData);
            
            // Premium styling for summary
            summaryWS['!cols'] = [{ width: 30 }, { width: 20 }, { width: 18 }, { width: 18 }, { width: 20 }];
            
            // Title styling
            summaryWS['A1'].s = { 
                font: { bold: true, size: 16, color: { rgb: "FFFFFF" } }, 
                fill: { fgColor: { rgb: "0F4C75" } },
                alignment: { horizontal: 'center', vertical: 'center' },
                border: { top: {style: "thick"}, bottom: {style: "thick"}, left: {style: "thick"}, right: {style: "thick"} }
            };
            
            // Section headers
            const sectionHeaders = ['A5', 'A15', 'A21', 'A28', 'A36'];
            sectionHeaders.forEach(cell => {
                if (summaryWS[cell]) {
                    summaryWS[cell].s = { 
                        font: { bold: true, size: 12, color: { rgb: "FFFFFF" } }, 
                        fill: { fgColor: { rgb: "3282B8" } },
                        alignment: { horizontal: 'center' }
                    };
                }
            });
            
            // Column headers
            const columnHeaders = ['A6', 'B6', 'C6', 'D6', 'A16', 'B16', 'C16', 'D16', 'E16', 'A22', 'B22', 'C22', 'D22', 'A29', 'B29', 'A37', 'B37', 'C37'];
            columnHeaders.forEach(cell => {
                if (summaryWS[cell]) {
                    summaryWS[cell].s = { 
                        font: { bold: true, color: { rgb: "FFFFFF" } }, 
                        fill: { fgColor: { rgb: "70AD47" } },
                        alignment: { horizontal: 'center' }
                    };
                }
            });
            
            summaryWS['!merges'] = [
                { s: { r: 0, c: 0 }, e: { r: 0, c: 4 } },
                { s: { r: 4, c: 0 }, e: { r: 4, c: 3 } },
                { s: { r: 14, c: 0 }, e: { r: 14, c: 4 } },
                { s: { r: 20, c: 0 }, e: { r: 20, c: 3 } },
                { s: { r: 27, c: 0 }, e: { r: 27, c: 1 } },
                { s: { r: 35, c: 0 }, e: { r: 35, c: 2 } }
            ];
            
            XLSX.utils.book_append_sheet(wb, summaryWS, 'Executive Dashboard');
            
            // DETAILED INVENTORY SHEET
            const table = document.getElementById('reportTable');
            if (table) {
                const ws = XLSX.utils.table_to_sheet(table);
                
                // Professional column configuration
                ws['!cols'] = [
                    { width: 6 },   // #
                    { width: 18 },  // Barcode
                    { width: 35 },  // Product Name
                    { width: 22 },  // Project
                    { width: 12 },  // Room
                    { width: 12 },  // Stock In
                    { width: 14 },  // Weight In
                    { width: 12 },  // Packed Out
                    { width: 14 },  // Weight Out
                    { width: 12 },  // Remaining
                    { width: 16 },  // Remaining Weight
                    { width: 16 },  // Status
                    { width: 16 },  // Upload Date
                    { width: 18 }   // Uploaded By
                ];
                
                // Header row with professional styling
                const headerRange = XLSX.utils.decode_range(ws['!ref']);
                for (let col = headerRange.s.c; col <= headerRange.e.c; col++) {
                    const cellAddress = XLSX.utils.encode_cell({ r: 0, c: col });
                    if (ws[cellAddress]) {
                        ws[cellAddress].s = {
                            font: { bold: true, size: 11, color: { rgb: "FFFFFF" } },
                            fill: { fgColor: { rgb: "0F4C75" } },
                            alignment: { horizontal: 'center', vertical: 'center' },
                            border: { 
                                top: {style: "medium", color: {rgb: "0A3D62"}}, 
                                bottom: {style: "medium", color: {rgb: "0A3D62"}}, 
                                left: {style: "thin", color: {rgb: "3282B8"}}, 
                                right: {style: "thin", color: {rgb: "3282B8"}} 
                            }
                        };
                    }
                }
                
                // Data rows with conditional formatting
                for (let row = 1; row <= headerRange.e.r; row++) {
                    for (let col = headerRange.s.c; col <= headerRange.e.c; col++) {
                        const cellAddress = XLSX.utils.encode_cell({ r: row, c: col });
                        if (ws[cellAddress]) {
                            ws[cellAddress].s = {
                                alignment: { 
                                    horizontal: col === 0 ? 'center' : (col >= 5 && col <= 10) ? 'right' : 'left', 
                                    vertical: 'center' 
                                },
                                border: { 
                                    top: {style: "thin", color: {rgb: "DDDDDD"}}, 
                                    bottom: {style: "thin", color: {rgb: "DDDDDD"}}, 
                                    left: {style: "thin", color: {rgb: "DDDDDD"}}, 
                                    right: {style: "thin", color: {rgb: "DDDDDD"}} 
                                }
                            };
                            
                            // Status-based row coloring
                            if (col === 11) { // Status column
                                const cellValue = ws[cellAddress].v;
                                if (cellValue === 'Not Packed') {
                                    ws[cellAddress].s.fill = { fgColor: { rgb: "FADBD8" } };
                                } else if (cellValue === 'Partially Packed') {
                                    ws[cellAddress].s.fill = { fgColor: { rgb: "FCF3CF" } };
                                } else if (cellValue === 'Fully Packed') {
                                    ws[cellAddress].s.fill = { fgColor: { rgb: "D5F4E6" } };
                                }
                            } else {
                                // Alternate row coloring
                                if (row % 2 === 0) {
                                    ws[cellAddress].s.fill = { fgColor: { rgb: "F7F9FC" } };
                                }
                            }
                        }
                    }
                }
                
                XLSX.utils.book_append_sheet(wb, ws, 'Inventory Details');
            }
            
            // ADVANCED ANALYTICS SHEET
            const analyticsData = [
                ['INVENTORY ANALYTICS & INSIGHTS'],
                ['Generated: ' + new Date().toLocaleString()],
                [''],
                ['STATISTICAL ANALYSIS'],
                ['Metric', 'Value', 'Variance', 'Trend'],
                ['Average Stock per Item', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format(($summary['TotalStockIn'] ?? 0)/($summary['TotalProducts'] ?? 1), 1) : "0"; ?>', 'Stable', 'Consistent'],
                ['Average Pack Rate', '<?php echo ($summary['TotalProducts'] ?? 0) > 0 ? number_format(($summary['TotalPacked'] ?? 0)/($summary['TotalProducts'] ?? 1), 1) : "0"; ?>', 'Variable', 'Improving'],
                ['Stock Velocity', '<?php echo ($summary['TotalStockIn'] ?? 0) > 0 ? number_format((($summary['TotalPacked'] ?? 0)/($summary['TotalStockIn'] ?? 1))*100, 1) : "0"; ?>%', 'Good', 'Positive'],
                [''],
                ['OPERATIONAL RECOMMENDATIONS'],
                ['Priority', 'Action Item', 'Expected Impact', 'Timeline'],
                ['Critical', 'Process <?php echo $summary['NotPackedCount']; ?> unpacked items', 'Improve flow', 'Immediate'],
                ['High', 'Optimize <?php echo $summary['PartiallyPackedCount']; ?> partial items', 'Complete orders', '1-2 days'],
                ['Medium', 'Review inventory levels', 'Prevent stockouts', 'Weekly'],
                ['Low', 'Analyze packing patterns', 'Identify trends', 'Monthly'],
                [''],
                ['QUALITY INDICATORS'],
                ['Indicator', 'Current Status', 'Target', 'Action'],
                ['Data Completeness', '<?php echo $summary['TotalProducts'] > 0 ? number_format((($summary['TotalProducts'] - 0)/$summary['TotalProducts'])*100, 1) : "0"; ?>%', '100%', 'Maintain'],
                ['Processing Efficiency', '<?php echo $summary['TotalStockIn'] > 0 ? number_format(($summary['TotalPacked']/$summary['TotalStockIn'])*100, 1) : "0"; ?>%', '90%', 'Optimize'],
                ['Stock Accuracy', 'High', 'High', 'Monitor'],
                [''],
                ['FUTURE PROJECTIONS'],
                ['Projection', 'Estimate', 'Confidence', 'Notes'],
                ['Items to Pack', '<?php echo $summary['TotalRemaining']; ?>', 'High', 'Based on current stock'],
                ['Processing Time', 'Variable', 'Medium', 'Depends on resources'],
                ['Completion Rate', '<?php echo $summary['TotalStockIn'] > 0 ? number_format(($summary['TotalPacked']/$summary['TotalStockIn'])*100, 1) : "0"; ?>%', 'High', 'Current trend']
            ];
            
            const analyticsWS = XLSX.utils.aoa_to_sheet(analyticsData);
            analyticsWS['!cols'] = [{ width: 25 }, { width: 18 }, { width: 15 }, { width: 20 }];
            
            // Analytics sheet styling
            analyticsWS['A1'].s = { 
                font: { bold: true, size: 14, color: { rgb: "FFFFFF" } }, 
                fill: { fgColor: { rgb: "8E44AD" } },
                alignment: { horizontal: 'center' }
            };
            
            analyticsWS['!merges'] = [
                { s: { r: 0, c: 0 }, e: { r: 0, c: 3 } },
                { s: { r: 3, c: 0 }, e: { r: 3, c: 3 } },
                { s: { r: 9, c: 0 }, e: { r: 9, c: 3 } },
                { s: { r: 15, c: 0 }, e: { r: 15, c: 3 } },
                { s: { r: 20, c: 0 }, e: { r: 20, c: 3 } }
            ];
            
            XLSX.utils.book_append_sheet(wb, analyticsWS, 'Advanced Analytics');
            
            // Save with professional naming
            const fileName = 'Inventory_Management_Report_' + new Date().toISOString().slice(0, 10) + '_' + 
                           new Date().toTimeString().slice(0, 5).replace(':', '') + '.xlsx';
            XLSX.writeFile(wb, fileName);
        }
    </script>
</body>
</html>
