<?php
session_start();
include_once('../includes/config.php');

// Helper function to safely escape HTML content
function safe_htmlspecialchars($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Check if user is logged in
if (!isset($_SESSION['uid'])) {
    header('location:../login.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// Get date range for reports
$dateFrom = $_GET['date_from'] ?? date('Y-m-01'); // First day of current month
$dateTo = $_GET['date_to'] ?? date('Y-m-d'); // Today
$customerId = $_GET['customer_id'] ?? '';

// Build WHERE conditions
$whereConditions = [];
$whereConditions[] = "ca.TransactionDate BETWEEN '$dateFrom' AND '$dateTo'";

if (!empty($customerId)) {
    $whereConditions[] = "ca.CustomerID = " . intval($customerId);
}

$whereClause = "WHERE " . implode(" AND ", $whereConditions);

// Summary Statistics
$summaryQuery = "SELECT 
    COUNT(DISTINCT c.ID) as total_customers,
    COUNT(ca.ID) as total_transactions,
    SUM(CASE WHEN ca.TransactionType = 'Invoice' THEN ca.Amount ELSE 0 END) as total_invoices,
    SUM(CASE WHEN ca.TransactionType = 'Payment' THEN ca.Amount ELSE 0 END) as total_payments,
    SUM(CASE WHEN ca.TransactionType = 'Credit Note' THEN ca.Amount ELSE 0 END) as total_credits,
    SUM(CASE WHEN ca.TransactionType = 'Debit Note' THEN ca.Amount ELSE 0 END) as total_debits,
    SUM(ca.BalanceAmount) as total_outstanding
    FROM tblcustomer_accounts ca 
    JOIN tblcustomers c ON ca.CustomerID = c.ID 
    $whereClause";

$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Customer-wise Report
$customerReportQuery = "SELECT 
    c.CustomerCode,
    c.CustomerName,
    c.CompanyName,
    COUNT(ca.ID) as transaction_count,
    SUM(CASE WHEN ca.TransactionType = 'Invoice' THEN ca.Amount ELSE 0 END) as total_invoices,
    SUM(CASE WHEN ca.TransactionType = 'Payment' THEN ca.Amount ELSE 0 END) as total_payments,
    SUM(ca.BalanceAmount) as outstanding_balance,
    MAX(ca.TransactionDate) as last_transaction_date
    FROM tblcustomers c
    LEFT JOIN tblcustomer_accounts ca ON c.ID = ca.CustomerID AND ca.TransactionDate BETWEEN '$dateFrom' AND '$dateTo'
    " . (empty($customerId) ? "" : "WHERE c.ID = " . intval($customerId)) . "
    GROUP BY c.ID, c.CustomerCode, c.CustomerName, c.CompanyName
    HAVING transaction_count > 0 OR '" . $customerId . "' != ''
    ORDER BY outstanding_balance DESC, c.CustomerName";

$customerReportResult = mysqli_query($con, $customerReportQuery);

// Monthly Trend Data
$monthlyQuery = "SELECT 
    DATE_FORMAT(ca.TransactionDate, '%Y-%m') as month,
    SUM(CASE WHEN ca.TransactionType = 'Invoice' THEN ca.Amount ELSE 0 END) as invoices,
    SUM(CASE WHEN ca.TransactionType = 'Payment' THEN ca.Amount ELSE 0 END) as payments
    FROM tblcustomer_accounts ca 
    JOIN tblcustomers c ON ca.CustomerID = c.ID 
    $whereClause
    GROUP BY DATE_FORMAT(ca.TransactionDate, '%Y-%m')
    ORDER BY month DESC
    LIMIT 12";

$monthlyResult = mysqli_query($con, $monthlyQuery);

// Aging Report (Outstanding invoices by age)
$agingQuery = "SELECT 
    CASE 
        WHEN DATEDIFF(CURDATE(), ca.DueDate) <= 0 THEN 'Current'
        WHEN DATEDIFF(CURDATE(), ca.DueDate) <= 30 THEN '1-30 Days'
        WHEN DATEDIFF(CURDATE(), ca.DueDate) <= 60 THEN '31-60 Days'
        WHEN DATEDIFF(CURDATE(), ca.DueDate) <= 90 THEN '61-90 Days'
        ELSE '90+ Days'
    END as aging_bucket,
    COUNT(ca.ID) as invoice_count,
    SUM(ca.BalanceAmount) as total_amount
    FROM tblcustomer_accounts ca 
    JOIN tblcustomers c ON ca.CustomerID = c.ID 
    WHERE ca.TransactionType = 'Invoice' 
    AND ca.Status IN ('Pending', 'Partial')
    AND ca.BalanceAmount > 0
    " . (empty($customerId) ? "" : "AND ca.CustomerID = " . intval($customerId)) . "
    GROUP BY aging_bucket
    ORDER BY 
        CASE aging_bucket
            WHEN 'Current' THEN 1
            WHEN '1-30 Days' THEN 2
            WHEN '31-60 Days' THEN 3
            WHEN '61-90 Days' THEN 4
            WHEN '90+ Days' THEN 5
        END";

$agingResult = mysqli_query($con, $agingQuery);

// Get customers for dropdown
$customersQuery = "SELECT ID, CustomerCode, CustomerName FROM tblcustomers WHERE IsActive = 1 ORDER BY CustomerName";
$customersResult = mysqli_query($con, $customersQuery);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Customer Reports - Ihita</title>
    
    <!-- Custom fonts for this template-->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css?family=Nunito:200,200i,300,300i,400,400i,600,600i,700,700i,800,800i,900,900i" rel="stylesheet">
    
    <!-- Custom styles for this template-->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
    
    <!-- Chart.js -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>

<body id="page-top">
    <!-- Page Wrapper -->
    <div id="wrapper">
        <!-- Content Wrapper -->
        <div id="content-wrapper" class="d-flex flex-column" style="margin-left: 0 !important;">
            <!-- Main Content -->
            <div id="content">
                <!-- Topbar -->
                <?php include_once('includes/customer-topbar.php'); ?>
                <!-- End of Topbar -->

                <!-- Begin Page Content -->
                <div class="container-fluid">
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">Customer Reports</h1>
                        <div>
                            <button class="btn btn-sm btn-primary" onclick="window.print()">
                                <i class="fas fa-print fa-sm text-white-50"></i> Print Report
                            </button>
                            <button class="btn btn-sm btn-success" onclick="exportToCSV()">
                                <i class="fas fa-download fa-sm text-white-50"></i> Export CSV
                            </button>
                        </div>
                    </div>

                    <!-- Filters -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Report Filters</h6>
                        </div>
                        <div class="card-body">
                            <form method="GET" class="row">
                                <div class="col-md-3">
                                    <label>Customer</label>
                                    <select name="customer_id" class="form-control">
                                        <option value="">All Customers</option>
                                        <?php while ($customer = mysqli_fetch_assoc($customersResult)): ?>
                                            <option value="<?php echo $customer['ID']; ?>" 
                                                    <?php echo ($customerId == $customer['ID']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($customer['CustomerCode'] . ' - ' . $customer['CustomerName']); ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label>From Date</label>
                                    <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                </div>
                                <div class="col-md-3">
                                    <label>To Date</label>
                                    <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                </div>
                                <div class="col-md-3">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">Generate Report</button>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-primary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Customers</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $summary['total_customers']; ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-users fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-success shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Invoices</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($summary['total_invoices'], 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-file-invoice fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-info shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Total Payments</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($summary['total_payments'], 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-money-check fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-warning shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Outstanding</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($summary['total_outstanding'], 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Charts Row -->
                    <div class="row mb-4">
                        <div class="col-xl-8 col-lg-7">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Monthly Trend</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="monthlyChart"></canvas>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-4 col-lg-5">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Aging Analysis</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="agingChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Customer-wise Report -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Customer-wise Report</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered" id="customerTable" width="100%" cellspacing="0">
                                    <thead>
                                        <tr>
                                            <th>Customer Code</th>
                                            <th>Customer Name</th>
                                            <th>Company</th>
                                            <th>Transactions</th>
                                            <th>Total Invoices</th>
                                            <th>Total Payments</th>
                                            <th>Outstanding</th>
                                            <th>Last Transaction</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($row = mysqli_fetch_assoc($customerReportResult)): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($row['CustomerCode']); ?></td>
                                                <td><?php echo htmlspecialchars($row['CustomerName']); ?></td>
                                                <td><?php echo htmlspecialchars($row['CompanyName']); ?></td>
                                                <td><?php echo $row['transaction_count']; ?></td>
                                                <td>₹<?php echo number_format($row['total_invoices'], 2); ?></td>
                                                <td>₹<?php echo number_format($row['total_payments'], 2); ?></td>
                                                <td>₹<?php echo number_format($row['outstanding_balance'], 2); ?></td>
                                                <td><?php echo $row['last_transaction_date'] ? date('M d, Y', strtotime($row['last_transaction_date'])) : '-'; ?></td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Aging Report -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Outstanding Aging Report</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered" width="100%" cellspacing="0">
                                    <thead>
                                        <tr>
                                            <th>Aging Bucket</th>
                                            <th>Number of Invoices</th>
                                            <th>Total Amount</th>
                                            <th>Percentage</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        $totalAging = 0;
                                        $agingData = [];
                                        mysqli_data_seek($agingResult, 0);
                                        while ($row = mysqli_fetch_assoc($agingResult)) {
                                            $totalAging += $row['total_amount'];
                                            $agingData[] = $row;
                                        }
                                        
                                        foreach ($agingData as $row): 
                                            $percentage = $totalAging > 0 ? ($row['total_amount'] / $totalAging) * 100 : 0;
                                        ?>
                                            <tr>
                                                <td><?php echo $row['aging_bucket']; ?></td>
                                                <td><?php echo $row['invoice_count']; ?></td>
                                                <td>₹<?php echo number_format($row['total_amount'], 2); ?></td>
                                                <td><?php echo number_format($percentage, 1); ?>%</td>
                                            </tr>
                                        <?php endforeach; ?>
                                        <tr class="font-weight-bold">
                                            <td>Total</td>
                                            <td><?php echo array_sum(array_column($agingData, 'invoice_count')); ?></td>
                                            <td>₹<?php echo number_format($totalAging, 2); ?></td>
                                            <td>100%</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- /.container-fluid -->
            </div>
            <!-- End of Main Content -->

            <!-- Footer -->
            <?php include_once('includes/customer-footer.php'); ?>
            <!-- End of Footer -->
        </div>
        <!-- End of Content Wrapper -->
    </div>
    <!-- End of Page Wrapper -->

    <!-- Bootstrap core JavaScript-->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>

    <!-- Core plugin JavaScript-->
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>

    <!-- Page level plugins -->
    <script src="../vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../vendor/datatables/dataTables.bootstrap4.min.js"></script>

    <!-- Custom scripts for all pages-->
    <script src="../js/sb-admin-2.min.js"></script>

    <script>
        $(document).ready(function() {
            $('#customerTable').DataTable();
            
            // Monthly Chart
            var monthlyData = <?php echo json_encode(array_reverse(mysqli_fetch_all($monthlyResult, MYSQLI_ASSOC))); ?>;
            var monthlyLabels = monthlyData.map(item => item.month);
            var invoiceData = monthlyData.map(item => parseFloat(item.invoices));
            var paymentData = monthlyData.map(item => parseFloat(item.payments));
            
            var ctx1 = document.getElementById('monthlyChart').getContext('2d');
            var monthlyChart = new Chart(ctx1, {
                type: 'line',
                data: {
                    labels: monthlyLabels,
                    datasets: [{
                        label: 'Invoices',
                        data: invoiceData,
                        borderColor: 'rgb(54, 162, 235)',
                        backgroundColor: 'rgba(54, 162, 235, 0.1)',
                        tension: 0.1
                    }, {
                        label: 'Payments',
                        data: paymentData,
                        borderColor: 'rgb(75, 192, 192)',
                        backgroundColor: 'rgba(75, 192, 192, 0.1)',
                        tension: 0.1
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return '₹' + value.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
            
            // Aging Chart
            var agingData = <?php echo json_encode($agingData); ?>;
            var agingLabels = agingData.map(item => item.aging_bucket);
            var agingAmounts = agingData.map(item => parseFloat(item.total_amount));
            
            var ctx2 = document.getElementById('agingChart').getContext('2d');
            var agingChart = new Chart(ctx2, {
                type: 'doughnut',
                data: {
                    labels: agingLabels,
                    datasets: [{
                        data: agingAmounts,
                        backgroundColor: [
                            '#36b9cc',
                            '#1cc88a',
                            '#f6c23e',
                            '#e74a3b',
                            '#6f42c1'
                        ]
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.label + ': ₹' + context.parsed.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
        });
        
        function exportToCSV() {
            var csv = [];
            var rows = document.querySelectorAll("#customerTable tr");
            
            for (var i = 0; i < rows.length; i++) {
                var row = [], cols = rows[i].querySelectorAll("td, th");
                
                for (var j = 0; j < cols.length; j++) {
                    row.push('"' + cols[j].innerText.replace(/"/g, '""') + '"');
                }
                
                csv.push(row.join(","));
            }
            
            var csvFile = new Blob([csv.join("\n")], {type: "text/csv"});
            var downloadLink = document.createElement("a");
            downloadLink.download = "customer_report_<?php echo date('Y-m-d'); ?>.csv";
            downloadLink.href = window.URL.createObjectURL(csvFile);
            downloadLink.style.display = "none";
            document.body.appendChild(downloadLink);
            downloadLink.click();
            document.body.removeChild(downloadLink);
        }
    </script>
</body>
</html>
