<?php
session_start();
include_once('../includes/config.php');

// Helper function to safely escape HTML content
function safe_htmlspecialchars($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Check if user is logged in
if (!isset($_SESSION['uid'])) {
    header('location:../login.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// Check if tables exist before running queries
$tablesExist = true;
$missingTables = [];

$checkTables = [
    'tblcustomers' => 'SELECT 1 FROM tblcustomers LIMIT 1',
    'tblcustomer_accounts' => 'SELECT 1 FROM tblcustomer_accounts LIMIT 1'
];

foreach ($checkTables as $tableName => $checkQuery) {
    $result = @mysqli_query($con, $checkQuery);
    if (!$result) {
        $tablesExist = false;
        $missingTables[] = $tableName;
    }
}

if (!$tablesExist) {
    // Show setup message if tables don't exist
    $setupMsg = 'Customer module tables are missing: ' . implode(', ', $missingTables) . '. Please run the database setup script.';
    
    // Create tables automatically
    $createCustomersTable = "CREATE TABLE IF NOT EXISTS `tblcustomers` (
        `ID` int(11) NOT NULL AUTO_INCREMENT,
        `CustomerCode` varchar(50) NOT NULL UNIQUE,
        `CustomerName` varchar(255) NOT NULL,
        `CompanyName` varchar(255) DEFAULT NULL,
        `Email` varchar(120) DEFAULT NULL,
        `Phone` varchar(20) DEFAULT NULL,
        `Address` text DEFAULT NULL,
        `City` varchar(100) DEFAULT NULL,
        `State` varchar(100) DEFAULT NULL,
        `Country` varchar(100) DEFAULT NULL,
        `PostalCode` varchar(20) DEFAULT NULL,
        `TaxNumber` varchar(50) DEFAULT NULL,
        `CreditLimit` decimal(12,2) DEFAULT 0.00,
        `CurrentBalance` decimal(12,2) DEFAULT 0.00,
        `IsActive` tinyint(1) DEFAULT 1,
        `AddedBy` int(11) NOT NULL,
        `CreatedDate` timestamp NOT NULL DEFAULT current_timestamp(),
        `UpdatedDate` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
        PRIMARY KEY (`ID`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
    
    $createAccountsTable = "CREATE TABLE IF NOT EXISTS `tblcustomer_accounts` (
        `ID` int(11) NOT NULL AUTO_INCREMENT,
        `CustomerID` int(11) NOT NULL,
        `InvoiceNumber` varchar(100) NOT NULL,
        `TransactionType` enum('Invoice','Payment','Credit Note','Debit Note') NOT NULL,
        `TransactionDate` date NOT NULL,
        `DueDate` date DEFAULT NULL,
        `Amount` decimal(12,2) NOT NULL,
        `PaidAmount` decimal(12,2) DEFAULT 0.00,
        `BalanceAmount` decimal(12,2) NOT NULL,
        `Description` text DEFAULT NULL,
        `PaymentMode` varchar(50) DEFAULT NULL,
        `ReferenceNumber` varchar(100) DEFAULT NULL,
        `TaxAmount` decimal(10,2) DEFAULT 0.00,
        `Status` enum('Pending','Partial','Paid','Overdue') DEFAULT 'Pending',
        `AttachmentPath` varchar(255) DEFAULT NULL,
        `AddedBy` int(11) NOT NULL,
        `CreatedDate` timestamp NOT NULL DEFAULT current_timestamp(),
        `UpdatedDate` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
        PRIMARY KEY (`ID`),
        KEY `CustomerID` (`CustomerID`),
        FOREIGN KEY (`CustomerID`) REFERENCES `tblcustomers` (`ID`) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
    
    mysqli_query($con, $createCustomersTable);
    mysqli_query($con, $createAccountsTable);
    
    // Refresh the page after creating tables
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit();
}

// Get dashboard statistics
$totalCustomers = 0;
$activeCustomers = 0;
$totalOutstanding = 0;
$totalOverdue = 0;

$statsQuery = "SELECT 
    COUNT(*) as total_customers,
    SUM(CASE WHEN IsActive = 1 THEN 1 ELSE 0 END) as active_customers,
    SUM(CurrentBalance) as total_outstanding
    FROM tblcustomers";
$statsResult = mysqli_query($con, $statsQuery);
if ($statsRow = mysqli_fetch_assoc($statsResult)) {
    $totalCustomers = $statsRow['total_customers'];
    $activeCustomers = $statsRow['active_customers'];
    $totalOutstanding = $statsRow['total_outstanding'] ?? 0;
}

// Get overdue amount
$overdueQuery = "SELECT SUM(ca.BalanceAmount) as overdue_amount
    FROM tblcustomer_accounts ca 
    WHERE ca.Status IN ('Pending', 'Partial') 
    AND ca.DueDate < CURDATE()
    AND ca.TransactionType = 'Invoice'";
$overdueResult = mysqli_query($con, $overdueQuery);
if ($overdueRow = mysqli_fetch_assoc($overdueResult)) {
    $totalOverdue = $overdueRow['overdue_amount'] ?? 0;
}

// Recent transactions
$recentQuery = "SELECT ca.*, c.CustomerName, c.CustomerCode 
    FROM tblcustomer_accounts ca 
    JOIN tblcustomers c ON ca.CustomerID = c.ID 
    ORDER BY ca.CreatedDate DESC 
    LIMIT 10";
$recentResult = mysqli_query($con, $recentQuery);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Customer Management - Ihita</title>
    
    <!-- Custom fonts for this template-->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css?family=Nunito:200,200i,300,300i,400,400i,600,600i,700,700i,800,800i,900,900i" rel="stylesheet">
    
    <!-- Custom styles for this template-->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/customer.css" rel="stylesheet">
</head>

<body id="page-top">
    <!-- Page Wrapper -->
    <div id="wrapper">
        <!-- Content Wrapper -->
        <div id="content-wrapper" class="d-flex flex-column" style="margin-left: 0 !important;">
            <!-- Main Content -->
            <div id="content">
                <!-- Topbar -->
                <?php include_once('includes/customer-topbar.php'); ?>
                <!-- End of Topbar -->

                <!-- Begin Page Content -->
                <div class="container-fluid">
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">Customer Management Dashboard</h1>
                        <div>
                            <a href="../dashboard.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm mr-2">
                                <i class="fas fa-home fa-sm text-white-50"></i> Main Dashboard
                            </a>
                            <a href="customers.php" class="d-none d-sm-inline-block btn btn-sm btn-primary shadow-sm">
                                <i class="fas fa-users fa-sm text-white-50"></i> Manage Customers
                            </a>
                        </div>
                    </div>

                    <!-- Content Row -->
                    <div class="row">
                        <!-- Total Customers Card -->
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-primary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Customers</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $totalCustomers; ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-users fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Active Customers Card -->
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-success shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Active Customers</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $activeCustomers; ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-user-check fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Total Outstanding Card -->
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-info shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Total Outstanding</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalOutstanding, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-rupee-sign fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Overdue Amount Card -->
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-warning shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Overdue Amount</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalOverdue, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Actions Row -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <div class="card shadow">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Quick Actions</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-md-3 mb-2">
                                            <a href="customers.php" class="btn btn-primary btn-block">
                                                <i class="fas fa-users"></i> Manage Customers
                                            </a>
                                        </div>
                                        <div class="col-md-3 mb-2">
                                            <a href="add-transaction.php" class="btn btn-success btn-block">
                                                <i class="fas fa-plus"></i> Add Transaction
                                            </a>
                                        </div>
                                        <div class="col-md-3 mb-2">
                                            <a href="accounts.php" class="btn btn-info btn-block">
                                                <i class="fas fa-file-invoice"></i> View Accounts
                                            </a>
                                        </div>
                                        <div class="col-md-3 mb-2">
                                            <a href="reports.php" class="btn btn-warning btn-block">
                                                <i class="fas fa-chart-bar"></i> Reports
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Recent Transactions -->
                    <div class="row">
                        <div class="col-12">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Recent Transactions</h6>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-bordered" width="100%" cellspacing="0">
                                            <thead>
                                                <tr>
                                                    <th>Date</th>
                                                    <th>Customer</th>
                                                    <th>Invoice #</th>
                                                    <th>Type</th>
                                                    <th>Amount</th>
                                                    <th>Balance</th>
                                                    <th>Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php if (mysqli_num_rows($recentResult) > 0): ?>
                                                    <?php while ($row = mysqli_fetch_assoc($recentResult)): ?>
                                                        <tr>
                                                            <td><?php echo date('M d, Y', strtotime($row['TransactionDate'])); ?></td>
                                                            <td><?php echo htmlspecialchars($row['CustomerName']); ?></td>
                                                            <td><?php echo htmlspecialchars($row['InvoiceNumber']); ?></td>
                                                            <td>
                                                                <span class="badge badge-<?php 
                                                                    echo $row['TransactionType'] == 'Invoice' ? 'primary' : 
                                                                        ($row['TransactionType'] == 'Payment' ? 'success' : 'info'); 
                                                                ?>">
                                                                    <?php echo $row['TransactionType']; ?>
                                                                </span>
                                                            </td>
                                                            <td>₹<?php echo number_format($row['Amount'], 2); ?></td>
                                                            <td>₹<?php echo number_format($row['BalanceAmount'], 2); ?></td>
                                                            <td>
                                                                <span class="badge badge-<?php 
                                                                    echo $row['Status'] == 'Paid' ? 'success' : 
                                                                        ($row['Status'] == 'Overdue' ? 'danger' : 'warning'); 
                                                                ?>">
                                                                    <?php echo $row['Status']; ?>
                                                                </span>
                                                            </td>
                                                        </tr>
                                                    <?php endwhile; ?>
                                                <?php else: ?>
                                                    <tr>
                                                        <td colspan="7" class="text-center">No transactions found</td>
                                                    </tr>
                                                <?php endif; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- /.container-fluid -->
            </div>
            <!-- End of Main Content -->

            <!-- Footer -->
            <?php include_once('includes/customer-footer.php'); ?>
            <!-- End of Footer -->
        </div>
        <!-- End of Content Wrapper -->
    </div>
    <!-- End of Page Wrapper -->

    <!-- Bootstrap core JavaScript-->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>

    <!-- Core plugin JavaScript-->
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>

    <!-- Custom scripts for all pages-->
    <script src="../js/sb-admin-2.min.js"></script>
</body>
</html>
