-- Customer Management Module Database Tables
-- Run this SQL script to manually create the customer module tables

-- Customer Master Table
CREATE TABLE IF NOT EXISTS `tblcustomers` (
    `ID` int(11) NOT NULL AUTO_INCREMENT,
    `CustomerCode` varchar(50) NOT NULL UNIQUE,
    `CustomerName` varchar(255) NOT NULL,
    `CompanyName` varchar(255) DEFAULT NULL,
    `Email` varchar(120) DEFAULT NULL,
    `Phone` varchar(20) DEFAULT NULL,
    `Address` text DEFAULT NULL,
    `City` varchar(100) DEFAULT NULL,
    `State` varchar(100) DEFAULT NULL,
    `Country` varchar(100) DEFAULT NULL,
    `PostalCode` varchar(20) DEFAULT NULL,
    `TaxNumber` varchar(50) DEFAULT NULL,
    `CreditLimit` decimal(12,2) DEFAULT 0.00,
    `CurrentBalance` decimal(12,2) DEFAULT 0.00,
    `IsActive` tinyint(1) DEFAULT 1,
    `AddedBy` int(11) NOT NULL,
    `CreatedDate` timestamp NOT NULL DEFAULT current_timestamp(),
    `UpdatedDate` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
    PRIMARY KEY (`ID`),
    KEY `IsActive` (`IsActive`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Customer Accounts/Transactions Table
CREATE TABLE IF NOT EXISTS `tblcustomer_accounts` (
    `ID` int(11) NOT NULL AUTO_INCREMENT,
    `CustomerID` int(11) NOT NULL,
    `InvoiceNumber` varchar(100) NOT NULL,
    `TransactionType` enum('Invoice','Payment','Credit Note','Debit Note') NOT NULL,
    `TransactionDate` date NOT NULL,
    `DueDate` date DEFAULT NULL,
    `Amount` decimal(12,2) NOT NULL,
    `PaidAmount` decimal(12,2) DEFAULT 0.00,
    `BalanceAmount` decimal(12,2) NOT NULL,
    `Description` text DEFAULT NULL,
    `PaymentMode` varchar(50) DEFAULT NULL,
    `ReferenceNumber` varchar(100) DEFAULT NULL,
    `TaxAmount` decimal(10,2) DEFAULT 0.00,
    `Status` enum('Pending','Partial','Paid','Overdue') DEFAULT 'Pending',
    `AttachmentPath` varchar(255) DEFAULT NULL,
    `AddedBy` int(11) NOT NULL,
    `CreatedDate` timestamp NOT NULL DEFAULT current_timestamp(),
    `UpdatedDate` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
    PRIMARY KEY (`ID`),
    KEY `CustomerID` (`CustomerID`),
    KEY `TransactionType` (`TransactionType`),
    KEY `Status` (`Status`),
    KEY `TransactionDate` (`TransactionDate`),
    KEY `DueDate` (`DueDate`),
    FOREIGN KEY (`CustomerID`) REFERENCES `tblcustomers` (`ID`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Sample Data for Testing (Optional)
INSERT IGNORE INTO `tblcustomers` (`CustomerCode`, `CustomerName`, `CompanyName`, `Email`, `Phone`, `Address`, `City`, `State`, `Country`, `CreditLimit`, `AddedBy`) VALUES
('CUST001', 'John Smith', 'ABC Technologies', 'john@abctech.com', '9876543210', '123 Business Street', 'Mumbai', 'Maharashtra', 'India', 50000.00, 1),
('CUST002', 'Sarah Johnson', 'XYZ Solutions', 'sarah@xyzsol.com', '9876543211', '456 Corporate Ave', 'Delhi', 'Delhi', 'India', 75000.00, 1),
('CUST003', 'Michael Brown', 'Tech Innovations', 'mike@techinno.com', '9876543212', '789 Innovation Hub', 'Bangalore', 'Karnataka', 'India', 100000.00, 1);

-- Sample Transactions (Optional)
INSERT IGNORE INTO `tblcustomer_accounts` (`CustomerID`, `InvoiceNumber`, `TransactionType`, `TransactionDate`, `DueDate`, `Amount`, `BalanceAmount`, `Description`, `AddedBy`) VALUES
((SELECT ID FROM tblcustomers WHERE CustomerCode = 'CUST001'), 'INV-001', 'Invoice', CURDATE(), DATE_ADD(CURDATE(), INTERVAL 30 DAY), 25000.00, 25000.00, 'Website development project', 1),
((SELECT ID FROM tblcustomers WHERE CustomerCode = 'CUST002'), 'INV-002', 'Invoice', CURDATE(), DATE_ADD(CURDATE(), INTERVAL 30 DAY), 35000.00, 35000.00, 'Mobile app development', 1),
((SELECT ID FROM tblcustomers WHERE CustomerCode = 'CUST001'), 'PAY-001', 'Payment', CURDATE(), NULL, 10000.00, 0.00, 'Partial payment for INV-001', 1);

-- Update customer balances based on transactions
UPDATE tblcustomers c SET CurrentBalance = (
    SELECT COALESCE(SUM(
        CASE 
            WHEN ca.TransactionType IN ('Invoice', 'Debit Note') THEN ca.BalanceAmount
            ELSE -ca.BalanceAmount
        END
    ), 0)
    FROM tblcustomer_accounts ca 
    WHERE ca.CustomerID = c.ID
);

-- Create indexes for better performance
CREATE INDEX idx_customer_accounts_date ON tblcustomer_accounts(TransactionDate);
CREATE INDEX idx_customer_accounts_status ON tblcustomer_accounts(Status);
CREATE INDEX idx_customers_active ON tblcustomers(IsActive);
CREATE INDEX idx_customer_accounts_balance ON tblcustomer_accounts(BalanceAmount);
