<?php
session_start();
include_once('../includes/config.php');

// Check if user is admin
if (!isset($_SESSION['uid']) || $_SESSION['usertype'] != 'admin') {
    header('location:../unauthorized.php');
    exit();
}

// Get filter parameters
$filterType = isset($_GET['type']) ? $_GET['type'] : '';
$filterDate = isset($_GET['date']) ? $_GET['date'] : '';
$filterMonth = isset($_GET['month']) ? $_GET['month'] : date('Y-m');

// Build query with filters
$whereClause = "WHERE 1=1";
$params = [];
$types = "";

if ($filterType && in_array($filterType, ['Income', 'Expense'])) {
    $whereClause .= " AND t.TransactionType = ?";
    $params[] = $filterType;
    $types .= "s";
}

if ($filterDate) {
    $whereClause .= " AND t.TransactionDate = ?";
    $params[] = $filterDate;
    $types .= "s";
} elseif ($filterMonth) {
    $whereClause .= " AND DATE_FORMAT(t.TransactionDate, '%Y-%m') = ?";
    $params[] = $filterMonth;
    $types .= "s";
}

// Get transactions with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$recordsPerPage = 20;
$offset = ($page - 1) * $recordsPerPage;

$countQuery = "SELECT COUNT(*) as total FROM tblfinance_transactions t $whereClause";
if ($types) {
    $stmt = mysqli_prepare($con, $countQuery);
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $totalResult = mysqli_stmt_get_result($stmt);
} else {
    $totalResult = mysqli_query($con, $countQuery);
}
$totalRecords = mysqli_fetch_assoc($totalResult)['total'];
$totalPages = ceil($totalRecords / $recordsPerPage);

$transactionsQuery = "SELECT t.*, a.AccountName 
                     FROM tblfinance_transactions t
                     LEFT JOIN tblfinance_accounts a ON t.AccountID = a.ID
                     $whereClause
                     ORDER BY t.TransactionDate DESC, t.CreatedDate DESC 
                     LIMIT $recordsPerPage OFFSET $offset";

if ($types) {
    $stmt = mysqli_prepare($con, $transactionsQuery);
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $transactionsResult = mysqli_stmt_get_result($stmt);
} else {
    $transactionsResult = mysqli_query($con, $transactionsQuery);
}

// Get summary statistics for current filter
$summaryQuery = "SELECT 
                    SUM(CASE WHEN TransactionType = 'Income' THEN Amount ELSE 0 END) as total_income,
                    SUM(CASE WHEN TransactionType = 'Expense' THEN Amount ELSE 0 END) as total_expense,
                    COUNT(*) as total_transactions
                 FROM tblfinance_transactions t $whereClause";

if ($types) {
    $stmt = mysqli_prepare($con, $summaryQuery);
    mysqli_stmt_bind_param($stmt, $types, ...$params);
    mysqli_stmt_execute($stmt);
    $summaryResult = mysqli_stmt_get_result($stmt);
} else {
    $summaryResult = mysqli_query($con, $summaryQuery);
}
$summary = mysqli_fetch_assoc($summaryResult);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>All Transactions - Finance Management</title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
    <link href="css/finance.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="finance-container">
        <!-- Header Section -->
        <div class="finance-header">
            <div class="container-fluid">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h1><i class="fas fa-list"></i> All Transactions</h1>
                        <div class="subtitle">View and manage your financial transactions</div>
                    </div>
                    <div class="col-md-4 text-right">
                        <a href="index.php" class="btn btn-light">
                            <i class="fas fa-arrow-left"></i> Back to Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filter and Quick Actions -->
        <div class="row mb-4">
            <div class="col-lg-12">
                <div class="quick-actions slide-in">
                    <div class="row">
                        <div class="col-md-8">
                            <h6><i class="fas fa-filter"></i> Filter Transactions</h6>
                            <form method="GET" class="form-inline">
                                <select name="type" class="form-control mr-2 mb-2">
                                    <option value="">All Types</option>
                                    <option value="Income" <?php echo $filterType == 'Income' ? 'selected' : ''; ?>>Income</option>
                                    <option value="Expense" <?php echo $filterType == 'Expense' ? 'selected' : ''; ?>>Expense</option>
                                </select>
                                <input type="month" name="month" class="form-control mr-2 mb-2" value="<?php echo $filterMonth; ?>">
                                <input type="date" name="date" class="form-control mr-2 mb-2" value="<?php echo $filterDate; ?>" placeholder="Specific Date">
                                <button type="submit" class="btn btn-primary mr-2 mb-2">
                                    <i class="fas fa-search"></i> Filter
                                </button>
                                <a href="transactions.php" class="btn btn-secondary mb-2">
                                    <i class="fas fa-times"></i> Clear
                                </a>
                            </form>
                        </div>
                        <div class="col-md-4 text-right">
                            <a href="add-transaction.php?type=income" class="btn btn-success action-btn mb-2">
                                <i class="fas fa-plus"></i> Add Income
                            </a>
                            <a href="add-transaction.php?type=expense" class="btn btn-danger action-btn mb-2">
                                <i class="fas fa-minus"></i> Add Expense
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Summary Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card finance-card income-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Total Income</div>
                        <div class="stat-value">₹<?php echo number_format($summary['total_income'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card expense-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Total Expense</div>
                        <div class="stat-value">₹<?php echo number_format($summary['total_expense'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card net-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Net Amount</div>
                        <div class="stat-value">₹<?php echo number_format($summary['total_income'] - $summary['total_expense'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card balance-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Total Transactions</div>
                        <div class="stat-value"><?php echo $summary['total_transactions']; ?></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Transactions Table -->
        <div class="row">
            <div class="col-lg-12">
                <div class="card finance-card slide-in">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">
                            <i class="fas fa-table"></i> Transactions List
                            <?php if ($filterType || $filterDate || $filterMonth): ?>
                                <span class="badge badge-info ml-2">Filtered</span>
                            <?php endif; ?>
                        </h6>
                    </div>
                    <div class="card-body p-0">
                        <?php if (mysqli_num_rows($transactionsResult) > 0): ?>
                            <div class="table-responsive">
                                <table class="table finance-table mb-0">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Type</th>
                                            <th>Description</th>
                                            <th>Account</th>
                                            <th>Category</th>
                                            <th>Payment Method</th>
                                            <th>Receipt</th>
                                            <th>Amount</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($transaction = mysqli_fetch_assoc($transactionsResult)): ?>
                                        <tr>
                                            <td>
                                                <div class="font-weight-bold">
                                                    <?php echo date('M d, Y', strtotime($transaction['TransactionDate'])); ?>
                                                </div>
                                                <div class="small text-muted">
                                                    <?php echo date('g:i A', strtotime($transaction['CreatedDate'])); ?>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo $transaction['TransactionType'] == 'Income' ? 'success' : 'danger'; ?> px-3 py-2">
                                                    <i class="fas fa-<?php echo $transaction['TransactionType'] == 'Income' ? 'arrow-up' : 'arrow-down'; ?>"></i>
                                                    <?php echo $transaction['TransactionType']; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="font-weight-bold"><?php echo htmlspecialchars($transaction['Description']); ?></div>
                                                <?php if ($transaction['Reference']): ?>
                                                <div class="small text-muted">Ref: <?php echo htmlspecialchars($transaction['Reference']); ?></div>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo htmlspecialchars($transaction['AccountName']); ?></td>
                                            <td>
                                                <?php if ($transaction['Category']): ?>
                                                    <span class="badge badge-secondary"><?php echo htmlspecialchars($transaction['Category']); ?></span>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <i class="fas fa-<?php 
                                                    switch($transaction['PaymentMethod']) {
                                                        case 'Cash': echo 'money-bill';
                                                        case 'Bank': echo 'university';
                                                        case 'Card': echo 'credit-card';
                                                        case 'Cheque': echo 'file-invoice';
                                                        case 'Online': echo 'globe';
                                                        default: echo 'question';
                                                    }
                                                ?>"></i>
                                                <?php echo $transaction['PaymentMethod']; ?>
                                            </td>
                                            <td class="text-center">
                                                <?php if ($transaction['ReceiptPhoto']): ?>
                                                    <button class="btn btn-sm btn-outline-info" onclick="viewReceipt('<?php echo htmlspecialchars($transaction['ReceiptPhoto']); ?>', '<?php echo htmlspecialchars($transaction['ReceiptOriginalName']); ?>')">
                                                        <i class="fas fa-file-image"></i>
                                                    </button>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-right">
                                                <div class="font-weight-bold <?php echo $transaction['TransactionType'] == 'Income' ? 'income-indicator' : 'expense-indicator'; ?>">
                                                    <?php echo $transaction['TransactionType'] == 'Income' ? '+' : '-'; ?>₹<?php echo number_format($transaction['Amount'], 2); ?>
                                                </div>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <button class="btn btn-outline-primary btn-sm" onclick="viewTransaction(<?php echo $transaction['ID']; ?>)">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                    <button class="btn btn-outline-warning btn-sm" onclick="editTransaction(<?php echo $transaction['ID']; ?>)">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button class="btn btn-outline-danger btn-sm" onclick="deleteTransaction(<?php echo $transaction['ID']; ?>)">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Pagination -->
                            <?php if ($totalPages > 1): ?>
                            <div class="card-footer">
                                <nav aria-label="Transactions pagination">
                                    <ul class="pagination justify-content-center mb-0">
                                        <?php if ($page > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo $page-1; ?>&type=<?php echo $filterType; ?>&date=<?php echo $filterDate; ?>&month=<?php echo $filterMonth; ?>">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                        
                                        <?php for ($i = max(1, $page-2); $i <= min($totalPages, $page+2); $i++): ?>
                                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&type=<?php echo $filterType; ?>&date=<?php echo $filterDate; ?>&month=<?php echo $filterMonth; ?>">
                                                    <?php echo $i; ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>
                                        
                                        <?php if ($page < $totalPages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo $page+1; ?>&type=<?php echo $filterType; ?>&date=<?php echo $filterDate; ?>&month=<?php echo $filterMonth; ?>">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                                <div class="text-center mt-2">
                                    <small class="text-muted">
                                        Showing <?php echo ($offset + 1); ?>-<?php echo min($offset + $recordsPerPage, $totalRecords); ?> of <?php echo $totalRecords; ?> transactions
                                    </small>
                                </div>
                            </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="p-5 text-center text-muted">
                                <i class="fas fa-receipt fa-4x mb-3"></i>
                                <h4>No Transactions Found</h4>
                                <p>No transactions match your current filter criteria.</p>
                                <div class="mt-3">
                                    <a href="add-transaction.php?type=income" class="btn btn-success action-btn mx-1">
                                        <i class="fas fa-plus"></i> Add Income
                                    </a>
                                    <a href="add-transaction.php?type=expense" class="btn btn-danger action-btn mx-1">
                                        <i class="fas fa-minus"></i> Add Expense
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Receipt Modal -->
    <div class="modal fade" id="receiptModal" tabindex="-1" role="dialog" aria-labelledby="receiptModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="receiptModalLabel">
                        <i class="fas fa-file-image"></i> Receipt Document
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body text-center">
                    <div id="receiptContent">
                        <!-- Receipt content will be loaded here -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="downloadReceipt">
                        <i class="fas fa-download"></i> Download
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    <script>
        function viewTransaction(id) {
            // You can implement a modal or redirect to view details
            alert('View transaction functionality can be implemented');
        }

        function editTransaction(id) {
            // You can implement edit functionality
            alert('Edit transaction functionality can be implemented');
        }

        function deleteTransaction(id) {
            if (confirm('Are you sure you want to delete this transaction? This action cannot be undone.')) {
                // Implement delete functionality via AJAX
                alert('Delete transaction functionality can be implemented');
            }
        }

        function viewReceipt(receiptPath, originalName) {
            const modal = $('#receiptModal');
            const content = $('#receiptContent');
            const downloadBtn = $('#downloadReceipt');
            
            // Clear previous content
            content.html('');
            
            // Check if it's a PDF or image
            const isPdf = receiptPath.toLowerCase().includes('.pdf');
            
            if (isPdf) {
                // For PDF files, show embed or link
                content.html(`
                    <div class="mb-3">
                        <i class="fas fa-file-pdf fa-4x text-danger"></i>
                        <h5 class="mt-2">${originalName}</h5>
                        <p class="text-muted">PDF Document</p>
                    </div>
                    <div>
                        <a href="${receiptPath}" target="_blank" class="btn btn-primary">
                            <i class="fas fa-external-link-alt"></i> Open PDF in New Tab
                        </a>
                    </div>
                `);
            } else {
                // For images, show preview
                content.html(`
                    <img src="${receiptPath}" alt="${originalName}" 
                         style="max-width: 100%; max-height: 500px; border: 1px solid #ddd; border-radius: 5px;">
                    <p class="mt-2 text-muted">${originalName}</p>
                `);
            }
            
            // Set download functionality
            downloadBtn.off('click').on('click', function() {
                const link = document.createElement('a');
                link.href = receiptPath;
                link.download = originalName;
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
            });
            
            modal.modal('show');
        }
    </script>
</body>
</html>
