<?php
session_start();
include_once('../includes/config.php');

// Check if user is admin
if (!isset($_SESSION['uid']) || $_SESSION['usertype'] != 'admin') {
    header('location:../unauthorized.php');
    exit();
}

// Get report parameters
$reportType = isset($_GET['report']) ? $_GET['report'] : 'monthly';
$reportMonth = isset($_GET['month']) ? $_GET['month'] : date('Y-m');
$reportYear = isset($_GET['year']) ? $_GET['year'] : date('Y');
$startDate = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$endDate = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Generate report data based on type
switch ($reportType) {
    case 'daily':
        $reportData = generateDailyReport($con, $reportMonth);
        break;
    case 'monthly':
        $reportData = generateMonthlyReport($con, $reportYear);
        break;
    case 'custom':
        $reportData = generateCustomReport($con, $startDate, $endDate);
        break;
    default:
        $reportData = generateMonthlyReport($con, $reportYear);
}

function generateDailyReport($con, $month) {
    $query = "SELECT 
                BalanceDate,
                OpeningBalance,
                TotalIncome,
                TotalExpense,
                ClosingBalance,
                (TotalIncome - TotalExpense) as NetAmount
              FROM tblfinance_daily_balances 
              WHERE DATE_FORMAT(BalanceDate, '%Y-%m') = '$month'
              ORDER BY BalanceDate DESC";
    
    $result = mysqli_query($con, $query);
    $data = [];
    $totalIncome = $totalExpense = 0;
    
    while ($row = mysqli_fetch_assoc($result)) {
        $data[] = $row;
        $totalIncome += $row['TotalIncome'];
        $totalExpense += $row['TotalExpense'];
    }
    
    return [
        'title' => 'Daily Report for ' . date('F Y', strtotime($month . '-01')),
        'data' => $data,
        'summary' => [
            'total_income' => $totalIncome,
            'total_expense' => $totalExpense,
            'net_amount' => $totalIncome - $totalExpense
        ]
    ];
}

function generateMonthlyReport($con, $year) {
    $query = "SELECT 
                DATE_FORMAT(BalanceDate, '%Y-%m') as month,
                SUM(TotalIncome) as monthly_income,
                SUM(TotalExpense) as monthly_expense,
                (SUM(TotalIncome) - SUM(TotalExpense)) as monthly_net
              FROM tblfinance_daily_balances 
              WHERE YEAR(BalanceDate) = '$year'
              GROUP BY DATE_FORMAT(BalanceDate, '%Y-%m')
              ORDER BY month DESC";
    
    $result = mysqli_query($con, $query);
    $data = [];
    $totalIncome = $totalExpense = 0;
    
    while ($row = mysqli_fetch_assoc($result)) {
        $data[] = $row;
        $totalIncome += $row['monthly_income'];
        $totalExpense += $row['monthly_expense'];
    }
    
    return [
        'title' => 'Monthly Report for ' . $year,
        'data' => $data,
        'summary' => [
            'total_income' => $totalIncome,
            'total_expense' => $totalExpense,
            'net_amount' => $totalIncome - $totalExpense
        ]
    ];
}

function generateCustomReport($con, $startDate, $endDate) {
    $query = "SELECT 
                BalanceDate,
                OpeningBalance,
                TotalIncome,
                TotalExpense,
                ClosingBalance,
                (TotalIncome - TotalExpense) as NetAmount
              FROM tblfinance_daily_balances 
              WHERE BalanceDate BETWEEN '$startDate' AND '$endDate'
              ORDER BY BalanceDate DESC";
    
    $result = mysqli_query($con, $query);
    $data = [];
    $totalIncome = $totalExpense = 0;
    
    while ($row = mysqli_fetch_assoc($result)) {
        $data[] = $row;
        $totalIncome += $row['TotalIncome'];
        $totalExpense += $row['TotalExpense'];
    }
    
    return [
        'title' => 'Custom Report from ' . date('M d, Y', strtotime($startDate)) . ' to ' . date('M d, Y', strtotime($endDate)),
        'data' => $data,
        'summary' => [
            'total_income' => $totalIncome,
            'total_expense' => $totalExpense,
            'net_amount' => $totalIncome - $totalExpense
        ]
    ];
}

// Get category-wise breakdown
$categoryQuery = "SELECT 
                    Category,
                    TransactionType,
                    SUM(Amount) as total_amount,
                    COUNT(*) as transaction_count
                  FROM tblfinance_transactions 
                  WHERE Category IS NOT NULL AND Category != ''
                  ";

if ($reportType == 'daily') {
    $categoryQuery .= " AND DATE_FORMAT(TransactionDate, '%Y-%m') = '$reportMonth'";
} elseif ($reportType == 'monthly') {
    $categoryQuery .= " AND YEAR(TransactionDate) = '$reportYear'";
} elseif ($reportType == 'custom') {
    $categoryQuery .= " AND TransactionDate BETWEEN '$startDate' AND '$endDate'";
}

$categoryQuery .= " GROUP BY Category, TransactionType ORDER BY total_amount DESC";
$categoryResult = mysqli_query($con, $categoryQuery);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Financial Reports - Finance Management</title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/finance.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <div class="finance-container">
        <!-- Header Section -->
        <div class="finance-header">
            <div class="container-fluid">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h1><i class="fas fa-chart-bar"></i> Financial Reports</h1>
                        <div class="subtitle">Analyze your financial performance</div>
                    </div>
                    <div class="col-md-4 text-right">
                        <a href="index.php" class="btn btn-light">
                            <i class="fas fa-arrow-left"></i> Back to Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Report Controls -->
        <div class="row mb-4">
            <div class="col-lg-12">
                <div class="quick-actions slide-in">
                    <div class="row">
                        <div class="col-md-8">
                            <h6><i class="fas fa-filter"></i> Report Options</h6>
                            <form method="GET" class="form-inline">
                                <select name="report" class="form-control mr-2 mb-2" onchange="toggleDateFields(this.value)">
                                    <option value="monthly" <?php echo $reportType == 'monthly' ? 'selected' : ''; ?>>Monthly Report</option>
                                    <option value="daily" <?php echo $reportType == 'daily' ? 'selected' : ''; ?>>Daily Report</option>
                                    <option value="custom" <?php echo $reportType == 'custom' ? 'selected' : ''; ?>>Custom Date Range</option>
                                </select>
                                
                                <div id="monthly-fields" style="<?php echo $reportType != 'monthly' ? 'display:none' : ''; ?>">
                                    <select name="year" class="form-control mr-2 mb-2">
                                        <?php for ($y = date('Y'); $y >= date('Y')-5; $y--): ?>
                                            <option value="<?php echo $y; ?>" <?php echo $reportYear == $y ? 'selected' : ''; ?>><?php echo $y; ?></option>
                                        <?php endfor; ?>
                                    </select>
                                </div>
                                
                                <div id="daily-fields" style="<?php echo $reportType != 'daily' ? 'display:none' : ''; ?>">
                                    <input type="month" name="month" class="form-control mr-2 mb-2" value="<?php echo $reportMonth; ?>">
                                </div>
                                
                                <div id="custom-fields" style="<?php echo $reportType != 'custom' ? 'display:none' : ''; ?>">
                                    <input type="date" name="start_date" class="form-control mr-2 mb-2" value="<?php echo $startDate; ?>" placeholder="Start Date">
                                    <input type="date" name="end_date" class="form-control mr-2 mb-2" value="<?php echo $endDate; ?>" placeholder="End Date">
                                </div>
                                
                                <button type="submit" class="btn btn-primary mr-2 mb-2">
                                    <i class="fas fa-chart-line"></i> Generate Report
                                </button>
                            </form>
                        </div>
                        <div class="col-md-4 text-right">
                            <button onclick="printReport()" class="btn btn-info action-btn mb-2">
                                <i class="fas fa-print"></i> Print Report
                            </button>
                            <button onclick="exportReport()" class="btn btn-success action-btn mb-2">
                                <i class="fas fa-download"></i> Export CSV
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Report Summary -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card finance-card income-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Total Income</div>
                        <div class="stat-value">₹<?php echo number_format($reportData['summary']['total_income'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card expense-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Total Expense</div>
                        <div class="stat-value">₹<?php echo number_format($reportData['summary']['total_expense'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card net-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Net Amount</div>
                        <div class="stat-value">₹<?php echo number_format($reportData['summary']['net_amount'], 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card balance-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Records</div>
                        <div class="stat-value"><?php echo count($reportData['data']); ?></div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Report Data Table -->
            <div class="col-lg-8">
                <div class="card finance-card slide-in">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">
                            <i class="fas fa-table"></i> <?php echo $reportData['title']; ?>
                        </h6>
                    </div>
                    <div class="card-body p-0">
                        <?php if (!empty($reportData['data'])): ?>
                            <div class="table-responsive">
                                <table class="table finance-table mb-0" id="reportTable">
                                    <thead>
                                        <tr>
                                            <?php if ($reportType == 'monthly'): ?>
                                                <th>Month</th>
                                            <?php else: ?>
                                                <th>Date</th>
                                                <th>Opening Balance</th>
                                            <?php endif; ?>
                                            <th>Income</th>
                                            <th>Expense</th>
                                            <th>Net Amount</th>
                                            <?php if ($reportType != 'monthly'): ?>
                                                <th>Closing Balance</th>
                                            <?php endif; ?>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($reportData['data'] as $row): ?>
                                        <tr>
                                            <?php if ($reportType == 'monthly'): ?>
                                                <td class="font-weight-bold">
                                                    <?php echo date('F Y', strtotime($row['month'] . '-01')); ?>
                                                </td>
                                                <td class="income-indicator text-right">
                                                    ₹<?php echo number_format($row['monthly_income'], 2); ?>
                                                </td>
                                                <td class="expense-indicator text-right">
                                                    ₹<?php echo number_format($row['monthly_expense'], 2); ?>
                                                </td>
                                                <td class="text-right font-weight-bold <?php echo $row['monthly_net'] >= 0 ? 'income-indicator' : 'expense-indicator'; ?>">
                                                    ₹<?php echo number_format($row['monthly_net'], 2); ?>
                                                </td>
                                            <?php else: ?>
                                                <td class="font-weight-bold">
                                                    <?php echo date('M d, Y', strtotime($row['BalanceDate'])); ?>
                                                </td>
                                                <td class="text-right">
                                                    ₹<?php echo number_format($row['OpeningBalance'], 2); ?>
                                                </td>
                                                <td class="income-indicator text-right">
                                                    ₹<?php echo number_format($row['TotalIncome'], 2); ?>
                                                </td>
                                                <td class="expense-indicator text-right">
                                                    ₹<?php echo number_format($row['TotalExpense'], 2); ?>
                                                </td>
                                                <td class="text-right font-weight-bold <?php echo $row['NetAmount'] >= 0 ? 'income-indicator' : 'expense-indicator'; ?>">
                                                    ₹<?php echo number_format($row['NetAmount'], 2); ?>
                                                </td>
                                                <td class="text-right font-weight-bold">
                                                    ₹<?php echo number_format($row['ClosingBalance'], 2); ?>
                                                </td>
                                            <?php endif; ?>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="p-4 text-center text-muted">
                                <i class="fas fa-chart-line fa-3x mb-3"></i>
                                <div>No data found for the selected period</div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Category Breakdown -->
            <div class="col-lg-4">
                <div class="card finance-card slide-in">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">
                            <i class="fas fa-chart-pie"></i> Category Breakdown
                        </h6>
                    </div>
                    <div class="card-body">
                        <?php if (mysqli_num_rows($categoryResult) > 0): ?>
                            <canvas id="categoryChart" width="300" height="300"></canvas>
                            
                            <div class="mt-4">
                                <h6>Category Details:</h6>
                                <?php 
                                mysqli_data_seek($categoryResult, 0);
                                while ($category = mysqli_fetch_assoc($categoryResult)): 
                                ?>
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <div>
                                        <span class="badge badge-<?php echo $category['TransactionType'] == 'Income' ? 'success' : 'danger'; ?>">
                                            <?php echo htmlspecialchars($category['Category']); ?>
                                        </span>
                                        <small class="text-muted">(<?php echo $category['transaction_count']; ?> transactions)</small>
                                    </div>
                                    <div class="font-weight-bold <?php echo $category['TransactionType'] == 'Income' ? 'income-indicator' : 'expense-indicator'; ?>">
                                        ₹<?php echo number_format($category['total_amount'], 0); ?>
                                    </div>
                                </div>
                                <?php endwhile; ?>
                            </div>
                        <?php else: ?>
                            <div class="text-center text-muted">
                                <i class="fas fa-chart-pie fa-3x mb-3"></i>
                                <div>No category data available</div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    
    <script>
        function toggleDateFields(reportType) {
            document.getElementById('monthly-fields').style.display = reportType === 'monthly' ? 'inline' : 'none';
            document.getElementById('daily-fields').style.display = reportType === 'daily' ? 'inline' : 'none';
            document.getElementById('custom-fields').style.display = reportType === 'custom' ? 'inline' : 'none';
        }

        function printReport() {
            window.print();
        }

        function exportReport() {
            // Convert table to CSV
            const table = document.getElementById('reportTable');
            let csv = '';
            
            // Add headers
            const headers = table.querySelectorAll('thead th');
            for (let i = 0; i < headers.length; i++) {
                csv += '"' + headers[i].textContent + '"';
                if (i < headers.length - 1) csv += ',';
            }
            csv += '\n';
            
            // Add data rows
            const rows = table.querySelectorAll('tbody tr');
            rows.forEach(row => {
                const cells = row.querySelectorAll('td');
                for (let i = 0; i < cells.length; i++) {
                    csv += '"' + cells[i].textContent.trim() + '"';
                    if (i < cells.length - 1) csv += ',';
                }
                csv += '\n';
            });
            
            // Download CSV
            const blob = new Blob([csv], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'financial_report_' + new Date().toISOString().slice(0, 10) + '.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }

        // Create category chart
        <?php if (mysqli_num_rows($categoryResult) > 0): ?>
        const ctx = document.getElementById('categoryChart').getContext('2d');
        const categoryData = {
            labels: [
                <?php 
                mysqli_data_seek($categoryResult, 0);
                $labels = [];
                $data = [];
                $colors = [];
                while ($category = mysqli_fetch_assoc($categoryResult)) {
                    $labels[] = "'" . addslashes($category['Category'] . ' (' . $category['TransactionType'] . ')') . "'";
                    $data[] = $category['total_amount'];
                    $colors[] = $category['TransactionType'] == 'Income' ? "'#1cc88a'" : "'#e74a3b'";
                }
                echo implode(', ', $labels);
                ?>
            ],
            datasets: [{
                data: [<?php echo implode(', ', $data); ?>],
                backgroundColor: [<?php echo implode(', ', $colors); ?>],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };

        new Chart(ctx, {
            type: 'doughnut',
            data: categoryData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });
        <?php endif; ?>
    </script>
</body>
</html>
