// Finance Module JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Initialize animations
    initializeAnimations();
    
    // Initialize tooltips
    initializeTooltips();
    
    // Initialize auto-save functionality
    initializeAutoSave();
    
    // Initialize keyboard shortcuts
    initializeKeyboardShortcuts();
});

function initializeAnimations() {
    // Fade in elements with delay
    const fadeElements = document.querySelectorAll('.fade-in');
    fadeElements.forEach((element, index) => {
        element.style.animationDelay = (index * 0.1) + 's';
    });
    
    // Slide in elements with delay
    const slideElements = document.querySelectorAll('.slide-in');
    slideElements.forEach((element, index) => {
        element.style.animationDelay = (index * 0.15) + 's';
    });
}

function initializeTooltips() {
    // Add tooltips to buttons and important elements
    const tooltipElements = document.querySelectorAll('[data-toggle="tooltip"]');
    tooltipElements.forEach(element => {
        if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
            new bootstrap.Tooltip(element);
        }
    });
}

function initializeAutoSave() {
    // Auto-save form data to localStorage
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        const inputs = form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            // Load saved data
            const savedValue = localStorage.getItem('finance_' + input.name);
            if (savedValue && input.type !== 'hidden') {
                input.value = savedValue;
            }
            
            // Save data on change
            input.addEventListener('change', function() {
                if (this.type !== 'hidden' && this.name) {
                    localStorage.setItem('finance_' + this.name, this.value);
                }
            });
        });
        
        // Clear saved data on successful submit
        form.addEventListener('submit', function() {
            setTimeout(() => {
                const inputs = this.querySelectorAll('input, select, textarea');
                inputs.forEach(input => {
                    if (input.name) {
                        localStorage.removeItem('finance_' + input.name);
                    }
                });
            }, 1000);
        });
    });
}

function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Ctrl/Cmd + I for Add Income
        if ((e.ctrlKey || e.metaKey) && e.key === 'i') {
            e.preventDefault();
            const incomeLink = document.querySelector('a[href*="add-transaction.php?type=income"]');
            if (incomeLink) {
                window.location.href = incomeLink.href;
            }
        }
        
        // Ctrl/Cmd + E for Add Expense
        if ((e.ctrlKey || e.metaKey) && e.key === 'e') {
            e.preventDefault();
            const expenseLink = document.querySelector('a[href*="add-transaction.php?type=expense"]');
            if (expenseLink) {
                window.location.href = expenseLink.href;
            }
        }
        
        // Ctrl/Cmd + R for Reports
        if ((e.ctrlKey || e.metaKey) && e.key === 'r') {
            e.preventDefault();
            const reportsLink = document.querySelector('a[href*="reports.php"]');
            if (reportsLink) {
                window.location.href = reportsLink.href;
            }
        }
        
        // Escape to go back to dashboard
        if (e.key === 'Escape') {
            const dashboardLink = document.querySelector('a[href*="index.php"]');
            if (dashboardLink && !document.querySelector('.modal.show')) {
                window.location.href = dashboardLink.href;
            }
        }
    });
}

// Utility Functions
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR',
        minimumFractionDigits: 2
    }).format(amount);
}

function showNotification(message, type = 'success') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
        ${message}
        <button type="button" class="close" data-dismiss="alert">
            <span>&times;</span>
        </button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 5000);
}

// Transaction Form Enhancements
if (document.querySelector('form[action*="add-transaction"]')) {
    enhanceTransactionForm();
}

function enhanceTransactionForm() {
    const form = document.querySelector('form');
    const amountField = document.querySelector('#amount');
    const descriptionField = document.querySelector('#description');
    const categoryField = document.querySelector('#category');
    
    // Amount field formatting
    if (amountField) {
        amountField.addEventListener('input', function() {
            // Remove any non-numeric characters except decimal point
            this.value = this.value.replace(/[^0-9.]/g, '');
            
            // Ensure only one decimal point
            const parts = this.value.split('.');
            if (parts.length > 2) {
                this.value = parts[0] + '.' + parts.slice(1).join('');
            }
            
            // Limit to 2 decimal places
            if (parts[1] && parts[1].length > 2) {
                this.value = parts[0] + '.' + parts[1].substring(0, 2);
            }
        });
    }
    
    // Description field auto-suggestions
    if (descriptionField && categoryField) {
        const commonDescriptions = {
            'Office Supplies': ['Stationery purchase', 'Printer paper', 'Office equipment'],
            'Marketing': ['Social media ads', 'Print advertisement', 'Website promotion'],
            'Utilities': ['Electricity bill', 'Internet bill', 'Water bill'],
            'Sales': ['Product sale', 'Service revenue', 'Consulting fee'],
            'Rent': ['Office rent', 'Equipment rental', 'Space rental']
        };
        
        categoryField.addEventListener('change', function() {
            const category = this.value;
            if (commonDescriptions[category]) {
                const datalist = document.getElementById('description-suggestions') || document.createElement('datalist');
                datalist.id = 'description-suggestions';
                datalist.innerHTML = '';
                
                commonDescriptions[category].forEach(desc => {
                    const option = document.createElement('option');
                    option.value = desc;
                    datalist.appendChild(option);
                });
                
                descriptionField.setAttribute('list', 'description-suggestions');
                document.body.appendChild(datalist);
            }
        });
    }
    
    // Form validation enhancement
    form.addEventListener('submit', function(e) {
        const amount = parseFloat(amountField.value);
        const description = descriptionField.value.trim();
        
        if (amount <= 0) {
            e.preventDefault();
            showNotification('Please enter a valid amount greater than 0', 'danger');
            amountField.focus();
            return;
        }
        
        if (description.length < 3) {
            e.preventDefault();
            showNotification('Please enter a description with at least 3 characters', 'danger');
            descriptionField.focus();
            return;
        }
        
        // Show loading state
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
        submitBtn.disabled = true;
        
        // Re-enable button after a delay (in case of errors)
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 10000);
    });
}

// Dashboard enhancements
if (document.querySelector('.finance-container')) {
    enhanceDashboard();
}

function enhanceDashboard() {
    // Animate numbers counting up
    const statValues = document.querySelectorAll('.stat-value');
    statValues.forEach(element => {
        const finalValue = element.textContent.replace(/[₹,]/g, '');
        if (!isNaN(finalValue) && finalValue !== '') {
            animateNumber(element, 0, parseFloat(finalValue), 1000);
        }
    });
    
    // Add real-time clock
    addRealTimeClock();
    
    // Add keyboard shortcut hints
    addKeyboardHints();
}

function animateNumber(element, start, end, duration) {
    const range = end - start;
    const increment = range / (duration / 16); // 60 FPS
    let current = start;
    const prefix = element.textContent.includes('₹') ? '₹' : '';
    
    const timer = setInterval(() => {
        current += increment;
        if (current >= end) {
            current = end;
            clearInterval(timer);
        }
        
        if (prefix === '₹') {
            element.textContent = prefix + Math.floor(current).toLocaleString('en-IN');
        } else {
            element.textContent = Math.floor(current);
        }
    }, 16);
}

function addRealTimeClock() {
    const clockElement = document.querySelector('.finance-header .col-md-4');
    if (clockElement) {
        function updateClock() {
            const now = new Date();
            const timeString = now.toLocaleTimeString('en-IN', {
                hour12: true,
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            
            const existingTime = clockElement.querySelector('.real-time-clock');
            if (existingTime) {
                existingTime.textContent = timeString;
            } else {
                const timeDiv = document.createElement('div');
                timeDiv.className = 'real-time-clock small';
                timeDiv.textContent = timeString;
                clockElement.appendChild(timeDiv);
            }
        }
        
        updateClock();
        setInterval(updateClock, 1000);
    }
}

function addKeyboardHints() {
    const hintsElement = document.createElement('div');
    hintsElement.className = 'keyboard-hints';
    hintsElement.innerHTML = `
        <div class="small text-muted mt-3">
            <strong>Keyboard Shortcuts:</strong>
            <span class="ml-2">Ctrl+I (Add Income)</span>
            <span class="ml-2">Ctrl+E (Add Expense)</span>
            <span class="ml-2">Ctrl+R (Reports)</span>
            <span class="ml-2">Esc (Dashboard)</span>
        </div>
    `;
    
    const container = document.querySelector('.finance-container');
    if (container) {
        container.appendChild(hintsElement);
    }
}

// Export functions for global use
window.FinanceModule = {
    formatCurrency,
    showNotification,
    animateNumber
};
