<?php
session_start();
include_once('../includes/config.php');

// Check if user is admin
if (!isset($_SESSION['uid']) || $_SESSION['usertype'] != 'admin') {
    header('location:../unauthorized.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// Check if tables exist before running queries
$tablesExist = true;
$missingTables = [];

$checkTables = [
    'tblfinance_accounts' => 'SELECT 1 FROM tblfinance_accounts LIMIT 1',
    'tblfinance_transactions' => 'SELECT 1 FROM tblfinance_transactions LIMIT 1', 
    'tblfinance_daily_balances' => 'SELECT 1 FROM tblfinance_daily_balances LIMIT 1'
];

foreach ($checkTables as $tableName => $checkQuery) {
    $result = @mysqli_query($con, $checkQuery);
    if (!$result) {
        $tablesExist = false;
        $missingTables[] = $tableName;
    }
}

if (!$tablesExist) {
    // Redirect to setup page if tables don't exist
    $setupUrl = '../create-finance-module-tables.php';
    $errorMsg = 'Finance module tables are missing: ' . implode(', ', $missingTables);
} else {
    // Get today's date
    $today = date('Y-m-d');
    
    // Get or create today's balance record
    $balanceQuery = "SELECT * FROM tblfinance_daily_balances WHERE BalanceDate = '$today'";
    $balanceResult = mysqli_query($con, $balanceQuery);
    
    if (mysqli_num_rows($balanceResult) == 0) {
        // Get yesterday's closing balance as today's opening balance
        $yesterdayQuery = "SELECT ClosingBalance FROM tblfinance_daily_balances 
                          WHERE BalanceDate < '$today' 
                          ORDER BY BalanceDate DESC LIMIT 1";
        $yesterdayResult = mysqli_query($con, $yesterdayQuery);
        $openingBalance = 0;
        
        if (mysqli_num_rows($yesterdayResult) > 0) {
            $openingBalance = mysqli_fetch_assoc($yesterdayResult)['ClosingBalance'];
        }
        
        // Create today's balance record
        $insertBalance = "INSERT INTO tblfinance_daily_balances (BalanceDate, OpeningBalance, ClosingBalance) 
                         VALUES ('$today', $openingBalance, $openingBalance)";
        mysqli_query($con, $insertBalance);
        
        // Fetch the newly created record
        $balanceResult = mysqli_query($con, $balanceQuery);
    }
    
    $todayBalance = mysqli_fetch_assoc($balanceResult);
    
    // Calculate today's income and expenses
    $todayIncomeQuery = "SELECT COALESCE(SUM(Amount), 0) as total_income 
                        FROM tblfinance_transactions 
                        WHERE TransactionType = 'Income' AND TransactionDate = '$today'";
    $todayIncomeResult = mysqli_query($con, $todayIncomeQuery);
    $todayIncome = mysqli_fetch_assoc($todayIncomeResult)['total_income'];
    
    $todayExpenseQuery = "SELECT COALESCE(SUM(Amount), 0) as total_expense 
                         FROM tblfinance_transactions 
                         WHERE TransactionType = 'Expense' AND TransactionDate = '$today'";
    $todayExpenseResult = mysqli_query($con, $todayExpenseQuery);
    $todayExpense = mysqli_fetch_assoc($todayExpenseResult)['total_expense'];
    
    // Calculate closing balance
    $closingBalance = $todayBalance['OpeningBalance'] + $todayIncome - $todayExpense;
    
    // Update today's balance record
    $updateBalanceQuery = "UPDATE tblfinance_daily_balances 
                          SET TotalIncome = $todayIncome, 
                              TotalExpense = $todayExpense, 
                              ClosingBalance = $closingBalance 
                          WHERE BalanceDate = '$today'";
    mysqli_query($con, $updateBalanceQuery);
    
    // Get monthly statistics
    $currentMonth = date('Y-m');
    $monthlyIncomeQuery = "SELECT COALESCE(SUM(Amount), 0) as monthly_income 
                          FROM tblfinance_transactions 
                          WHERE TransactionType = 'Income' AND DATE_FORMAT(TransactionDate, '%Y-%m') = '$currentMonth'";
    $monthlyIncomeResult = mysqli_query($con, $monthlyIncomeQuery);
    $monthlyIncome = mysqli_fetch_assoc($monthlyIncomeResult)['monthly_income'];
    
    $monthlyExpenseQuery = "SELECT COALESCE(SUM(Amount), 0) as monthly_expense 
                           FROM tblfinance_transactions 
                           WHERE TransactionType = 'Expense' AND DATE_FORMAT(TransactionDate, '%Y-%m') = '$currentMonth'";
    $monthlyExpenseResult = mysqli_query($con, $monthlyExpenseQuery);
    $monthlyExpense = mysqli_fetch_assoc($monthlyExpenseResult)['monthly_expense'];
    
    $monthlyNet = $monthlyIncome - $monthlyExpense;
    
    // Get recent transactions
    $recentTransactionsQuery = "SELECT t.*, a.AccountName 
                               FROM tblfinance_transactions t
                               LEFT JOIN tblfinance_accounts a ON t.AccountID = a.ID
                               ORDER BY t.CreatedDate DESC 
                               LIMIT 10";
    $recentTransactions = mysqli_query($con, $recentTransactionsQuery);
    
    // Get account summary
    $accountSummaryQuery = "SELECT 
                               COUNT(*) as total_accounts,
                               SUM(CASE WHEN AccountType = 'Income' THEN CurrentBalance ELSE 0 END) as total_income_accounts,
                               SUM(CASE WHEN AccountType = 'Expense' THEN CurrentBalance ELSE 0 END) as total_expense_accounts
                           FROM tblfinance_accounts WHERE Status = 'Active'";
    $accountSummaryResult = mysqli_query($con, $accountSummaryQuery);
    $accountSummary = mysqli_fetch_assoc($accountSummaryResult);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Finance Management - Minerva Technologies</title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/finance.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="finance-container">
        <!-- Header Section -->
        <div class="finance-header">
            <div class="container-fluid">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h1><i class="fas fa-chart-line"></i> Finance Management</h1>
                        <div class="subtitle">Manage your income, expenses, and financial reports</div>
                    </div>
                    <div class="col-md-4 text-right">
                        <div class="text-white">
                            <div class="small">Today: <?php echo date('M d, Y'); ?></div>
                            <div class="h5 mb-0"><?php echo date('l'); ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <?php if (!$tablesExist): ?>
            <!-- Setup Required Notice -->
            <div class="row">
                <div class="col-lg-12">
                    <div class="alert alert-warning alert-enhanced fade-in" role="alert">
                        <h4 class="alert-heading"><i class="fas fa-exclamation-triangle"></i> Database Setup Required</h4>
                        <p><?php echo $errorMsg; ?></p>
                        <hr>
                        <p class="mb-0">
                            Please run the database setup to create the required tables.
                            <a href="<?php echo $setupUrl; ?>" class="btn btn-warning ml-2">
                                <i class="fas fa-database"></i> Run Database Setup
                            </a>
                        </p>
                    </div>
                </div>
            </div>
        <?php else: ?>

        <!-- Balance Display -->
        <div class="row mb-4">
            <div class="col-lg-12">
                <div class="balance-display fade-in">
                    <div class="balance-label">Current Balance</div>
                    <div class="balance-amount">₹<?php echo number_format($closingBalance, 2); ?></div>
                    <div class="row mt-3">
                        <div class="col-md-4">
                            <div class="small">Opening Balance</div>
                            <div class="h5">₹<?php echo number_format($todayBalance['OpeningBalance'], 2); ?></div>
                        </div>
                        <div class="col-md-4">
                            <div class="small">Today's Income</div>
                            <div class="h5 text-success">+₹<?php echo number_format($todayIncome, 2); ?></div>
                        </div>
                        <div class="col-md-4">
                            <div class="small">Today's Expense</div>
                            <div class="h5 text-danger">-₹<?php echo number_format($todayExpense, 2); ?></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="row mb-4">
            <div class="col-lg-12">
                <div class="quick-actions slide-in">
                    <h5><i class="fas fa-bolt"></i> Quick Actions</h5>
                    <div class="row">
                        <div class="col-md-3 mb-2">
                            <a href="add-transaction.php?type=income" class="btn btn-add-income action-btn btn-block">
                                <i class="fas fa-plus-circle"></i> Add Income
                            </a>
                        </div>
                        <div class="col-md-3 mb-2">
                            <a href="add-transaction.php?type=expense" class="btn btn-add-expense action-btn btn-block">
                                <i class="fas fa-minus-circle"></i> Add Expense
                            </a>
                        </div>
                        <div class="col-md-3 mb-2">
                            <a href="reports.php" class="btn btn-view-reports action-btn btn-block">
                                <i class="fas fa-chart-bar"></i> View Reports
                            </a>
                        </div>
                        <div class="col-md-3 mb-2">
                            <a href="accounts.php" class="btn btn-manage-accounts action-btn btn-block">
                                <i class="fas fa-cogs"></i> Manage Accounts
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <!-- Monthly Income -->
            <div class="col-xl-3 col-md-6 mb-4">
                <div class="card finance-card income-card fade-in">
                    <div class="card-body">
                        <div class="row no-gutters align-items-center">
                            <div class="col mr-2">
                                <div class="stat-label">Monthly Income</div>
                                <div class="stat-value">₹<?php echo number_format($monthlyIncome, 0); ?></div>
                            </div>
                            <div class="col-auto">
                                <i class="fas fa-arrow-up stat-icon"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Monthly Expense -->
            <div class="col-xl-3 col-md-6 mb-4">
                <div class="card finance-card expense-card fade-in">
                    <div class="card-body">
                        <div class="row no-gutters align-items-center">
                            <div class="col mr-2">
                                <div class="stat-label">Monthly Expense</div>
                                <div class="stat-value">₹<?php echo number_format($monthlyExpense, 0); ?></div>
                            </div>
                            <div class="col-auto">
                                <i class="fas fa-arrow-down stat-icon"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Monthly Net -->
            <div class="col-xl-3 col-md-6 mb-4">
                <div class="card finance-card net-card fade-in">
                    <div class="card-body">
                        <div class="row no-gutters align-items-center">
                            <div class="col mr-2">
                                <div class="stat-label">Monthly Net</div>
                                <div class="stat-value">₹<?php echo number_format($monthlyNet, 0); ?></div>
                            </div>
                            <div class="col-auto">
                                <i class="fas fa-balance-scale stat-icon"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Total Accounts -->
            <div class="col-xl-3 col-md-6 mb-4">
                <div class="card finance-card balance-card fade-in">
                    <div class="card-body">
                        <div class="row no-gutters align-items-center">
                            <div class="col mr-2">
                                <div class="stat-label">Active Accounts</div>
                                <div class="stat-value"><?php echo $accountSummary['total_accounts']; ?></div>
                            </div>
                            <div class="col-auto">
                                <i class="fas fa-university stat-icon"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Transactions -->
        <div class="row">
            <div class="col-lg-12">
                <div class="card finance-card slide-in">
                    <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                        <h6 class="m-0 font-weight-bold text-primary">
                            <i class="fas fa-history"></i> Recent Transactions
                        </h6>
                        <a href="transactions.php" class="btn btn-sm btn-outline-primary">
                            <i class="fas fa-list"></i> View All
                        </a>
                    </div>
                    <div class="card-body p-0">
                        <?php if (mysqli_num_rows($recentTransactions) > 0): ?>
                            <div class="recent-transactions">
                                <?php while ($transaction = mysqli_fetch_assoc($recentTransactions)): ?>
                                <div class="transaction-item">
                                    <div class="row align-items-center">
                                        <div class="col-md-1">
                                            <i class="fas fa-<?php echo $transaction['TransactionType'] == 'Income' ? 'arrow-up text-success' : 'arrow-down text-danger'; ?> fa-2x"></i>
                                        </div>
                                        <div class="col-md-3">
                                            <div class="transaction-description"><?php echo htmlspecialchars($transaction['Description']); ?></div>
                                            <div class="transaction-date"><?php echo date('M d, Y', strtotime($transaction['TransactionDate'])); ?></div>
                                        </div>
                                        <div class="col-md-3">
                                            <div class="small text-muted">Account</div>
                                            <div><?php echo htmlspecialchars($transaction['AccountName']); ?></div>
                                        </div>
                                        <div class="col-md-2">
                                            <div class="small text-muted">Category</div>
                                            <div><?php echo htmlspecialchars($transaction['Category'] ?: 'N/A'); ?></div>
                                        </div>
                                        <div class="col-md-2">
                                            <div class="small text-muted">Method</div>
                                            <div><?php echo htmlspecialchars($transaction['PaymentMethod']); ?></div>
                                        </div>
                                        <div class="col-md-1 text-right">
                                            <div class="transaction-amount <?php echo $transaction['TransactionType'] == 'Income' ? 'income-indicator' : 'expense-indicator'; ?>">
                                                <?php echo $transaction['TransactionType'] == 'Income' ? '+' : '-'; ?>₹<?php echo number_format($transaction['Amount'], 2); ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endwhile; ?>
                            </div>
                        <?php else: ?>
                            <div class="p-4 text-center text-muted">
                                <i class="fas fa-receipt fa-3x mb-3"></i>
                                <div>No transactions found</div>
                                <div class="small">Start by adding your first income or expense transaction</div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <?php endif; ?>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    <script src="js/finance.js"></script>
</body>
</html>
