<?php
session_start();
include_once('../includes/config.php');

// Check if user is admin
if (!isset($_SESSION['uid']) || $_SESSION['usertype'] != 'admin') {
    header('location:../unauthorized.php');
    exit();
}

$type = isset($_GET['type']) ? $_GET['type'] : 'income';
$type = in_array($type, ['income', 'expense']) ? $type : 'income';

$message = '';
$error = '';

// Handle form submission
if (isset($_POST['action']) && $_POST['action'] === 'add_transaction') {
    $accountId = mysqli_real_escape_string($con, $_POST['account_id']);
    $transactionType = mysqli_real_escape_string($con, ucfirst($_POST['transaction_type']));
    $amount = floatval($_POST['amount']);
    $description = mysqli_real_escape_string($con, $_POST['description']);
    $transactionDate = mysqli_real_escape_string($con, $_POST['transaction_date']);
    $reference = mysqli_real_escape_string($con, $_POST['reference']);
    $category = mysqli_real_escape_string($con, $_POST['category']);
    $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
    $createdBy = $_SESSION['uid'];

    // Handle receipt upload
    $receiptPhoto = null;
    $receiptOriginalName = null;
    $receiptUploadDate = null;

    if (isset($_FILES['receipt_photo']) && $_FILES['receipt_photo']['error'] === UPLOAD_ERR_OK) {
        $uploadDir = 'uploads/receipts/';
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'application/pdf'];
        $maxFileSize = 5 * 1024 * 1024; // 5MB

        $fileType = $_FILES['receipt_photo']['type'];
        $fileSize = $_FILES['receipt_photo']['size'];
        $originalName = $_FILES['receipt_photo']['name'];

        if (in_array($fileType, $allowedTypes) && $fileSize <= $maxFileSize) {
            $fileExtension = pathinfo($originalName, PATHINFO_EXTENSION);
            $fileName = 'receipt_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.' . $fileExtension;
            $uploadPath = $uploadDir . $fileName;

            if (move_uploaded_file($_FILES['receipt_photo']['tmp_name'], $uploadPath)) {
                $receiptPhoto = $uploadPath;
                $receiptOriginalName = $originalName;
                $receiptUploadDate = date('Y-m-d H:i:s');
            } else {
                $error = "Failed to upload receipt file.";
            }
        } else {
            $error = "Invalid file type or file too large. Please upload images (JPG, PNG, GIF) or PDF files under 5MB.";
        }
    }

    if ($amount > 0 && !empty($description) && !empty($transactionDate) && empty($error)) {
        try {
            // Start transaction
            mysqli_autocommit($con, FALSE);

            // Insert transaction
            $insertTransaction = "INSERT INTO tblfinance_transactions 
                                (AccountID, TransactionType, Amount, Description, TransactionDate, Reference, Category, PaymentMethod, ReceiptPhoto, ReceiptOriginalName, ReceiptUploadDate, CreatedBy) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = mysqli_prepare($con, $insertTransaction);
            mysqli_stmt_bind_param($stmt, "isdssssssssi", $accountId, $transactionType, $amount, $description, $transactionDate, $reference, $category, $paymentMethod, $receiptPhoto, $receiptOriginalName, $receiptUploadDate, $createdBy);
            
            if (mysqli_stmt_execute($stmt)) {
                // Update account balance
                if ($transactionType == 'Income') {
                    $updateAccount = "UPDATE tblfinance_accounts SET CurrentBalance = CurrentBalance + ? WHERE ID = ?";
                } else {
                    $updateAccount = "UPDATE tblfinance_accounts SET CurrentBalance = CurrentBalance - ? WHERE ID = ?";
                }
                
                $stmt2 = mysqli_prepare($con, $updateAccount);
                mysqli_stmt_bind_param($stmt2, "di", $amount, $accountId);
                mysqli_stmt_execute($stmt2);

                // Update daily balance if transaction is for today
                $today = date('Y-m-d');
                if ($transactionDate === $today) {
                    // Get or create today's balance record
                    $checkBalance = "SELECT ID FROM tblfinance_daily_balances WHERE BalanceDate = ?";
                    $stmt3 = mysqli_prepare($con, $checkBalance);
                    mysqli_stmt_bind_param($stmt3, "s", $today);
                    mysqli_stmt_execute($stmt3);
                    $result = mysqli_stmt_get_result($stmt3);
                    
                    if (mysqli_num_rows($result) == 0) {
                        // Create today's balance record
                        $yesterdayQuery = "SELECT ClosingBalance FROM tblfinance_daily_balances 
                                          WHERE BalanceDate < ? ORDER BY BalanceDate DESC LIMIT 1";
                        $stmt4 = mysqli_prepare($con, $yesterdayQuery);
                        mysqli_stmt_bind_param($stmt4, "s", $today);
                        mysqli_stmt_execute($stmt4);
                        $yesterdayResult = mysqli_stmt_get_result($stmt4);
                        $openingBalance = 0;
                        
                        if (mysqli_num_rows($yesterdayResult) > 0) {
                            $openingBalance = mysqli_fetch_assoc($yesterdayResult)['ClosingBalance'];
                        }
                        
                        $insertBalance = "INSERT INTO tblfinance_daily_balances (BalanceDate, OpeningBalance, ClosingBalance) 
                                         VALUES (?, ?, ?)";
                        $stmt5 = mysqli_prepare($con, $insertBalance);
                        mysqli_stmt_bind_param($stmt5, "sdd", $today, $openingBalance, $openingBalance);
                        mysqli_stmt_execute($stmt5);
                    }
                    
                    // Recalculate today's totals
                    $recalculateQuery = "UPDATE tblfinance_daily_balances 
                                        SET TotalIncome = (SELECT COALESCE(SUM(Amount), 0) FROM tblfinance_transactions WHERE TransactionType = 'Income' AND TransactionDate = ?),
                                            TotalExpense = (SELECT COALESCE(SUM(Amount), 0) FROM tblfinance_transactions WHERE TransactionType = 'Expense' AND TransactionDate = ?),
                                            ClosingBalance = OpeningBalance + (SELECT COALESCE(SUM(Amount), 0) FROM tblfinance_transactions WHERE TransactionType = 'Income' AND TransactionDate = ?) - (SELECT COALESCE(SUM(Amount), 0) FROM tblfinance_transactions WHERE TransactionType = 'Expense' AND TransactionDate = ?)
                                        WHERE BalanceDate = ?";
                    $stmt6 = mysqli_prepare($con, $recalculateQuery);
                    mysqli_stmt_bind_param($stmt6, "sssss", $today, $today, $today, $today, $today);
                    mysqli_stmt_execute($stmt6);
                }

                mysqli_commit($con);
                $message = ucfirst($transactionType) . " transaction added successfully!";
            } else {
                throw new Exception("Error adding transaction: " . mysqli_error($con));
            }
        } catch (Exception $e) {
            mysqli_rollback($con);
            $error = $e->getMessage();
        }
        
        mysqli_autocommit($con, TRUE);
    } else {
        $error = "Please fill in all required fields with valid data.";
    }
}

// Get accounts for dropdown
$accountsQuery = "SELECT * FROM tblfinance_accounts WHERE Status = 'Active' ORDER BY AccountName";
$accountsResult = mysqli_query($con, $accountsQuery);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Add <?php echo ucfirst($type); ?> Transaction - Finance Management</title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/finance.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="finance-container">
        <!-- Header Section -->
        <div class="finance-header">
            <div class="container-fluid">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h1>
                            <i class="fas fa-<?php echo $type == 'income' ? 'plus-circle text-success' : 'minus-circle text-danger'; ?>"></i> 
                            Add <?php echo ucfirst($type); ?> Transaction
                        </h1>
                        <div class="subtitle">Record a new <?php echo $type; ?> transaction</div>
                    </div>
                    <div class="col-md-4 text-right">
                        <a href="index.php" class="btn btn-light">
                            <i class="fas fa-arrow-left"></i> Back to Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-success alert-enhanced fade-in" role="alert">
                <i class="fas fa-check-circle"></i> <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger alert-enhanced fade-in" role="alert">
                <i class="fas fa-exclamation-triangle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <!-- Transaction Form -->
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="finance-form fade-in">
                    <form method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="add_transaction">
                        <input type="hidden" name="transaction_type" value="<?php echo $type; ?>">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="account_id">
                                        <i class="fas fa-university"></i> Account *
                                    </label>
                                    <select class="form-control" id="account_id" name="account_id" required>
                                        <option value="">Select Account</option>
                                        <?php while ($account = mysqli_fetch_assoc($accountsResult)): ?>
                                            <option value="<?php echo $account['ID']; ?>">
                                                <?php echo htmlspecialchars($account['AccountName']); ?> 
                                                (<?php echo $account['AccountType']; ?>) - 
                                                ₹<?php echo number_format($account['CurrentBalance'], 2); ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="amount">
                                        <i class="fas fa-rupee-sign"></i> Amount *
                                    </label>
                                    <input type="number" step="0.01" min="0.01" class="form-control" id="amount" name="amount" required>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="description">
                                <i class="fas fa-comment"></i> Description *
                            </label>
                            <textarea class="form-control" id="description" name="description" rows="3" required placeholder="Enter transaction description..."></textarea>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="transaction_date">
                                        <i class="fas fa-calendar"></i> Transaction Date *
                                    </label>
                                    <input type="date" class="form-control" id="transaction_date" name="transaction_date" value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="payment_method">
                                        <i class="fas fa-credit-card"></i> Payment Method
                                    </label>
                                    <select class="form-control" id="payment_method" name="payment_method">
                                        <option value="Cash">Cash</option>
                                        <option value="Bank">Bank Transfer</option>
                                        <option value="Card">Credit/Debit Card</option>
                                        <option value="Cheque">Cheque</option>
                                        <option value="Online">Online Payment</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="category">
                                        <i class="fas fa-tag"></i> Category
                                    </label>
                                    <input type="text" class="form-control" id="category" name="category" placeholder="e.g., Office Supplies, Sales, etc.">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="reference">
                                        <i class="fas fa-barcode"></i> Reference
                                    </label>
                                    <input type="text" class="form-control" id="reference" name="reference" placeholder="Invoice number, receipt number, etc.">
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="receipt_photo">
                                <i class="fas fa-camera"></i> Receipt Photo/Document (Optional)
                            </label>
                            <input type="file" class="form-control-file" id="receipt_photo" name="receipt_photo" 
                                   accept="image/*,.pdf" onchange="previewReceipt(this)">
                            <small class="form-text text-muted">
                                Upload receipt image (JPG, PNG, GIF) or PDF document. Maximum size: 5MB
                            </small>
                            <div id="receipt-preview" class="mt-2" style="display: none;">
                                <img id="receipt-image" src="" alt="Receipt Preview" style="max-width: 200px; max-height: 200px; border: 1px solid #ddd; border-radius: 5px;">
                                <div id="receipt-info" class="small text-muted mt-1"></div>
                            </div>
                        </div>

                        <div class="form-group text-center mt-4">
                            <button type="submit" class="btn btn-<?php echo $type == 'income' ? 'success' : 'danger'; ?> action-btn px-5">
                                <i class="fas fa-save"></i> Add <?php echo ucfirst($type); ?> Transaction
                            </button>
                            <a href="index.php" class="btn btn-secondary action-btn px-5 ml-3">
                                <i class="fas fa-times"></i> Cancel
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Quick Add Another -->
        <?php if ($message): ?>
        <div class="row justify-content-center mt-4">
            <div class="col-lg-8 text-center">
                <div class="quick-actions slide-in">
                    <h6>Quick Actions</h6>
                    <a href="add-transaction.php?type=income" class="btn btn-success action-btn mx-2">
                        <i class="fas fa-plus"></i> Add Another Income
                    </a>
                    <a href="add-transaction.php?type=expense" class="btn btn-danger action-btn mx-2">
                        <i class="fas fa-minus"></i> Add Another Expense
                    </a>
                    <a href="transactions.php" class="btn btn-info action-btn mx-2">
                        <i class="fas fa-list"></i> View All Transactions
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    <script>
        // Auto-fill category based on transaction type
        document.addEventListener('DOMContentLoaded', function() {
            const categoryField = document.getElementById('category');
            const transactionType = '<?php echo $type; ?>';
            
            if (transactionType === 'income') {
                categoryField.placeholder = 'e.g., Sales, Service Revenue, Consulting...';
            } else {
                categoryField.placeholder = 'e.g., Office Supplies, Rent, Utilities...';
            }
        });

        // Receipt preview function
        function previewReceipt(input) {
            const preview = document.getElementById('receipt-preview');
            const image = document.getElementById('receipt-image');
            const info = document.getElementById('receipt-info');
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                const reader = new FileReader();
                
                // Validate file size (5MB)
                if (file.size > 5 * 1024 * 1024) {
                    alert('File size too large. Please select a file under 5MB.');
                    input.value = '';
                    preview.style.display = 'none';
                    return;
                }
                
                // Validate file type
                const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'application/pdf'];
                if (!allowedTypes.includes(file.type)) {
                    alert('Invalid file type. Please select an image (JPG, PNG, GIF) or PDF file.');
                    input.value = '';
                    preview.style.display = 'none';
                    return;
                }
                
                if (file.type === 'application/pdf') {
                    // For PDF files, show file info only
                    image.style.display = 'none';
                    info.innerHTML = `<i class="fas fa-file-pdf text-danger"></i> ${file.name} (${(file.size/1024/1024).toFixed(2)} MB)`;
                    preview.style.display = 'block';
                } else {
                    // For image files, show preview
                    reader.onload = function(e) {
                        image.src = e.target.result;
                        image.style.display = 'block';
                        info.innerHTML = `${file.name} (${(file.size/1024/1024).toFixed(2)} MB)`;
                        preview.style.display = 'block';
                    };
                    reader.readAsDataURL(file);
                }
            } else {
                preview.style.display = 'none';
            }
        }
    </script>
</body>
</html>
