<?php
session_start();
include_once('../includes/config.php');

// Check if user is admin
if (!isset($_SESSION['uid']) || $_SESSION['usertype'] != 'admin') {
    header('location:../unauthorized.php');
    exit();
}

$message = '';
$error = '';

// Handle account creation
if (isset($_POST['action']) && $_POST['action'] === 'add_account') {
    $accountName = mysqli_real_escape_string($con, $_POST['account_name']);
    $accountType = mysqli_real_escape_string($con, $_POST['account_type']);
    $openingBalance = floatval($_POST['opening_balance']);
    $description = mysqli_real_escape_string($con, $_POST['description']);

    if (!empty($accountName) && !empty($accountType)) {
        $insertAccount = "INSERT INTO tblfinance_accounts (AccountName, AccountType, OpeningBalance, CurrentBalance, Description) 
                         VALUES (?, ?, ?, ?, ?)";
        $stmt = mysqli_prepare($con, $insertAccount);
        mysqli_stmt_bind_param($stmt, "ssdds", $accountName, $accountType, $openingBalance, $openingBalance, $description);
        
        if (mysqli_stmt_execute($stmt)) {
            $message = "Account '$accountName' created successfully!";
        } else {
            $error = "Error creating account: " . mysqli_error($con);
        }
    } else {
        $error = "Account name and type are required.";
    }
}

// Handle account status update
if (isset($_POST['action']) && $_POST['action'] === 'update_status') {
    $accountId = intval($_POST['account_id']);
    $status = mysqli_real_escape_string($con, $_POST['status']);
    
    $updateStatus = "UPDATE tblfinance_accounts SET Status = ? WHERE ID = ?";
    $stmt = mysqli_prepare($con, $updateStatus);
    mysqli_stmt_bind_param($stmt, "si", $status, $accountId);
    
    if (mysqli_stmt_execute($stmt)) {
        $message = "Account status updated successfully!";
    } else {
        $error = "Error updating account status: " . mysqli_error($con);
    }
}

// Get all accounts
$accountsQuery = "SELECT a.*, 
                         COUNT(t.ID) as transaction_count,
                         SUM(CASE WHEN t.TransactionType = 'Income' THEN t.Amount ELSE 0 END) as total_income,
                         SUM(CASE WHEN t.TransactionType = 'Expense' THEN t.Amount ELSE 0 END) as total_expense
                  FROM tblfinance_accounts a
                  LEFT JOIN tblfinance_transactions t ON a.ID = t.AccountID
                  GROUP BY a.ID
                  ORDER BY a.AccountType, a.AccountName";
$accountsResult = mysqli_query($con, $accountsQuery);

// Get account type summary
$summaryQuery = "SELECT 
                    AccountType,
                    COUNT(*) as account_count,
                    SUM(CurrentBalance) as total_balance
                 FROM tblfinance_accounts 
                 WHERE Status = 'Active'
                 GROUP BY AccountType";
$summaryResult = mysqli_query($con, $summaryQuery);
$typeSummary = [];
while ($row = mysqli_fetch_assoc($summaryResult)) {
    $typeSummary[$row['AccountType']] = $row;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Manage Accounts - Finance Management</title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/finance.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@300;400;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="finance-container">
        <!-- Header Section -->
        <div class="finance-header">
            <div class="container-fluid">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h1><i class="fas fa-university"></i> Manage Accounts</h1>
                        <div class="subtitle">Create and manage your financial accounts</div>
                    </div>
                    <div class="col-md-4 text-right">
                        <a href="index.php" class="btn btn-light">
                            <i class="fas fa-arrow-left"></i> Back to Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Messages -->
        <?php if ($message): ?>
            <div class="alert alert-success alert-enhanced fade-in" role="alert">
                <i class="fas fa-check-circle"></i> <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger alert-enhanced fade-in" role="alert">
                <i class="fas fa-exclamation-triangle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <!-- Account Type Summary -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card finance-card income-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Income Accounts</div>
                        <div class="stat-value"><?php echo $typeSummary['Income']['account_count'] ?? 0; ?></div>
                        <div class="small">Balance: ₹<?php echo number_format($typeSummary['Income']['total_balance'] ?? 0, 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card expense-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Expense Accounts</div>
                        <div class="stat-value"><?php echo $typeSummary['Expense']['account_count'] ?? 0; ?></div>
                        <div class="small">Balance: ₹<?php echo number_format($typeSummary['Expense']['total_balance'] ?? 0, 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card balance-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Asset Accounts</div>
                        <div class="stat-value"><?php echo $typeSummary['Asset']['account_count'] ?? 0; ?></div>
                        <div class="small">Balance: ₹<?php echo number_format($typeSummary['Asset']['total_balance'] ?? 0, 2); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card finance-card net-card fade-in">
                    <div class="card-body text-center">
                        <div class="stat-label">Liability Accounts</div>
                        <div class="stat-value"><?php echo $typeSummary['Liability']['account_count'] ?? 0; ?></div>
                        <div class="small">Balance: ₹<?php echo number_format($typeSummary['Liability']['total_balance'] ?? 0, 2); ?></div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Add New Account Form -->
            <div class="col-lg-4">
                <div class="finance-form fade-in">
                    <h5 class="mb-4"><i class="fas fa-plus-circle"></i> Add New Account</h5>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_account">
                        
                        <div class="form-group">
                            <label for="account_name">
                                <i class="fas fa-tag"></i> Account Name *
                            </label>
                            <input type="text" class="form-control" id="account_name" name="account_name" required>
                        </div>

                        <div class="form-group">
                            <label for="account_type">
                                <i class="fas fa-list"></i> Account Type *
                            </label>
                            <select class="form-control" id="account_type" name="account_type" required>
                                <option value="">Select Type</option>
                                <option value="Income">Income</option>
                                <option value="Expense">Expense</option>
                                <option value="Asset">Asset</option>
                                <option value="Liability">Liability</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="opening_balance">
                                <i class="fas fa-rupee-sign"></i> Opening Balance
                            </label>
                            <input type="number" step="0.01" class="form-control" id="opening_balance" name="opening_balance" value="0.00">
                        </div>

                        <div class="form-group">
                            <label for="description">
                                <i class="fas fa-comment"></i> Description
                            </label>
                            <textarea class="form-control" id="description" name="description" rows="3" placeholder="Account description..."></textarea>
                        </div>

                        <button type="submit" class="btn btn-primary action-btn btn-block">
                            <i class="fas fa-save"></i> Create Account
                        </button>
                    </form>
                </div>
            </div>

            <!-- Accounts List -->
            <div class="col-lg-8">
                <div class="card finance-card slide-in">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">
                            <i class="fas fa-list"></i> All Accounts
                        </h6>
                    </div>
                    <div class="card-body p-0">
                        <?php if (mysqli_num_rows($accountsResult) > 0): ?>
                            <div class="table-responsive">
                                <table class="table finance-table mb-0">
                                    <thead>
                                        <tr>
                                            <th>Account Name</th>
                                            <th>Type</th>
                                            <th>Current Balance</th>
                                            <th>Transactions</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        $currentType = '';
                                        while ($account = mysqli_fetch_assoc($accountsResult)): 
                                            if ($currentType != $account['AccountType']) {
                                                $currentType = $account['AccountType'];
                                        ?>
                                        <tr class="table-secondary">
                                            <td colspan="6" class="font-weight-bold">
                                                <i class="fas fa-<?php 
                                                    switch($currentType) {
                                                        case 'Income': echo 'arrow-up';
                                                        case 'Expense': echo 'arrow-down';
                                                        case 'Asset': echo 'coins';
                                                        case 'Liability': echo 'credit-card';
                                                        default: echo 'question';
                                                    }
                                                ?>"></i>
                                                <?php echo $currentType; ?> Accounts
                                            </td>
                                        </tr>
                                        <?php } ?>
                                        <tr>
                                            <td>
                                                <div class="font-weight-bold"><?php echo htmlspecialchars($account['AccountName']); ?></div>
                                                <?php if ($account['Description']): ?>
                                                <div class="small text-muted"><?php echo htmlspecialchars($account['Description']); ?></div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php 
                                                    switch($account['AccountType']) {
                                                        case 'Income': echo 'success';
                                                        case 'Expense': echo 'danger';
                                                        case 'Asset': echo 'primary';
                                                        case 'Liability': echo 'warning';
                                                        default: echo 'secondary';
                                                    }
                                                ?>">
                                                    <?php echo $account['AccountType']; ?>
                                                </span>
                                            </td>
                                            <td class="text-right">
                                                <div class="font-weight-bold">₹<?php echo number_format($account['CurrentBalance'], 2); ?></div>
                                                <div class="small text-muted">
                                                    Opening: ₹<?php echo number_format($account['OpeningBalance'], 2); ?>
                                                </div>
                                            </td>
                                            <td class="text-center">
                                                <div class="font-weight-bold"><?php echo $account['transaction_count']; ?></div>
                                                <?php if ($account['transaction_count'] > 0): ?>
                                                <div class="small">
                                                    <span class="text-success">+₹<?php echo number_format($account['total_income'], 0); ?></span> /
                                                    <span class="text-danger">-₹<?php echo number_format($account['total_expense'], 0); ?></span>
                                                </div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="action" value="update_status">
                                                    <input type="hidden" name="account_id" value="<?php echo $account['ID']; ?>">
                                                    <select name="status" class="form-control form-control-sm" onchange="this.form.submit()">
                                                        <option value="Active" <?php echo $account['Status'] == 'Active' ? 'selected' : ''; ?>>Active</option>
                                                        <option value="Inactive" <?php echo $account['Status'] == 'Inactive' ? 'selected' : ''; ?>>Inactive</option>
                                                    </select>
                                                </form>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <button class="btn btn-outline-primary btn-sm" onclick="viewAccountDetails(<?php echo $account['ID']; ?>)">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                    <button class="btn btn-outline-warning btn-sm" onclick="editAccount(<?php echo $account['ID']; ?>)">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="p-4 text-center text-muted">
                                <i class="fas fa-university fa-3x mb-3"></i>
                                <div>No accounts found</div>
                                <div class="small">Create your first account to get started</div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    <script>
        function viewAccountDetails(id) {
            // Implement account details modal or page
            alert('Account details functionality can be implemented');
        }

        function editAccount(id) {
            // Implement edit account functionality
            alert('Edit account functionality can be implemented');
        }

        // Auto-suggest account names based on type
        document.getElementById('account_type').addEventListener('change', function() {
            const accountNameField = document.getElementById('account_name');
            const type = this.value;
            
            switch(type) {
                case 'Income':
                    accountNameField.placeholder = 'e.g., Sales Revenue, Service Income, Consulting Revenue';
                    break;
                case 'Expense':
                    accountNameField.placeholder = 'e.g., Office Rent, Utilities, Marketing Expenses';
                    break;
                case 'Asset':
                    accountNameField.placeholder = 'e.g., Cash Account, Bank Account, Inventory';
                    break;
                case 'Liability':
                    accountNameField.placeholder = 'e.g., Accounts Payable, Loans, Credit Cards';
                    break;
                default:
                    accountNameField.placeholder = 'Enter account name';
            }
        });
    </script>
</body>
</html>
