<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

// --- FILTERS ---
function getDateRange($filter) {
    $today = date('Y-m-d');
    switch ($filter) {
        case 'today':
            return [$today, $today];
        case 'yesterday':
            $yesterday = date('Y-m-d', strtotime('-1 day'));
            return [$yesterday, $yesterday];
        case 'this_week':
            $start = date('Y-m-d', strtotime('monday this week'));
            $end = date('Y-m-d', strtotime('sunday this week'));
            return [$start, $end];
        case 'this_month':
            $start = date('Y-m-01');
            $end = date('Y-m-t');
            return [$start, $end];
        case 'this_year':
            $start = date('Y-01-01');
            $end = date('Y-12-31');
            return [$start, $end];
        case 'custom':
            $start = isset($_GET['from']) ? $_GET['from'] : $today;
            $end = isset($_GET['to']) ? $_GET['to'] : $today;
            return [$start, $end];
        default:
            return [null, null]; // all time
    }
}

// --- Excel Export ---
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    $filter = isset($_GET['filter']) ? $_GET['filter'] : 'this_month';
    list($from, $to) = getDateRange($filter);
    $whereIncome = $whereExpense = "";
    if ($from && $to) {
        $whereIncome = "WHERE DATE(EntryDate) BETWEEN '$from' AND '$to'";
        $whereExpense = "WHERE DATE(EntryDate) BETWEEN '$from' AND '$to'";
    }
    
    // Payment Mode Filter
    $paymentMode = isset($_GET['payment_mode']) ? trim($_GET['payment_mode']) : '';
    if ($paymentMode !== '') {
        $paymentFilter = " AND PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
        $whereIncome = $whereIncome ? $whereIncome . $paymentFilter : "WHERE PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
        $whereExpense = $whereExpense ? $whereExpense . $paymentFilter : "WHERE PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
    }
    
    // Type Filter
    $typeFilter = isset($_GET['type_filter']) ? $_GET['type_filter'] : 'all';
    
    $orderBy = "ORDER BY CreatedAt DESC";
    $unionParts = [];
    
    if ($typeFilter == 'all' || $typeFilter == 'income') {
        $unionParts[] = "
            SELECT 
                'Income' AS Type,
                FundTitle AS Title,
                FundAmount AS Credit,
                NULL AS Debit,
                PaymentMode,
                FundDetails AS Details,
                u.FullName AS Staff,
                EntryDate AS CreatedAt,
                NULL AS VoucherImage
            FROM tblfunds
            LEFT JOIN tblusers u ON tblfunds.RaisedBy = u.ID
            $whereIncome
        ";
    }
    
    if ($typeFilter == 'all' || $typeFilter == 'expense') {
        $unionParts[] = "
            SELECT 
                'Expense' AS Type,
                ExpenseTitle AS Title,
                NULL AS Credit,
                ExpenseAmount AS Debit,
                PaymentMode,
                ExpenseDetails AS Details,
                u.FullName AS Staff,
                EntryDate AS CreatedAt,
                VoucherImage
            FROM tblexpenses
            LEFT JOIN tblusers u ON tblexpenses.UserID = u.ID
            $whereExpense
        ";
    }
    
    $unionQuery = implode(' UNION ALL ', $unionParts) . " $orderBy";
    $records = mysqli_query($con, $unionQuery);

    header("Content-Type: application/vnd.ms-excel");
    header("Content-Disposition: attachment; filename=income_expense_report.xls");
    echo "Type\tTitle\tCredit\tDebit\tPayment Mode\tDetails\tStaff\tDate\tVoucher\n";
    while($row = mysqli_fetch_assoc($records)) {
        echo $row['Type']."\t";
        echo $row['Title']."\t";
        echo ($row['Credit'] !== null ? $row['Credit'] : "")."\t";
        echo ($row['Debit'] !== null ? $row['Debit'] : "")."\t";
        echo $row['PaymentMode']."\t";
        echo $row['Details']."\t";
        echo $row['Staff']."\t";
        echo $row['CreatedAt']."\t";
        echo ($row['VoucherImage'] ? $row['VoucherImage'] : "")."\n";
    }
    exit();
}

// --- Default filter: today ---
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'today';
list($from, $to) = getDateRange($filter);

$whereIncome = $whereExpense = "";
if ($from && $to) {
    $whereIncome = "WHERE DATE(EntryDate) BETWEEN '$from' AND '$to'";
    $whereExpense = "WHERE DATE(EntryDate) BETWEEN '$from' AND '$to'";
}

// Payment Mode Filter
$paymentMode = isset($_GET['payment_mode']) ? trim($_GET['payment_mode']) : '';
if ($paymentMode !== '') {
    $paymentFilter = " AND PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
    $whereIncome = $whereIncome ? $whereIncome . $paymentFilter : "WHERE PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
    $whereExpense = $whereExpense ? $whereExpense . $paymentFilter : "WHERE PaymentMode = '" . mysqli_real_escape_string($con, $paymentMode) . "'";
}

// Type Filter
$typeFilter = isset($_GET['type_filter']) ? $_GET['type_filter'] : 'all';

// --- Sorting ---
$sortable = ['Type','Title','Credit','Debit','PaymentMode','Details','Staff','CreatedAt'];
$sort = isset($_GET['sort']) && in_array($_GET['sort'],$sortable) ? $_GET['sort'] : 'CreatedAt';
$order = (isset($_GET['order']) && strtolower($_GET['order'])=='asc') ? 'ASC' : 'DESC';

$orderBy = "ORDER BY $sort $order";

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// --- Search Filters for Admin ---
$searchTitle = isset($_GET['search_title']) ? trim($_GET['search_title']) : '';
$searchUser = isset($_GET['search_user']) ? trim($_GET['search_user']) : '';
$searchDetails = isset($_GET['search_details']) ? trim($_GET['search_details']) : '';

$searchIncome = $searchExpense = [];
if ($userType == 'admin') {
    if ($searchTitle !== '') {
        $searchIncome[] = "FundTitle LIKE '%" . mysqli_real_escape_string($con, $searchTitle) . "%'";
        $searchExpense[] = "ExpenseTitle LIKE '%" . mysqli_real_escape_string($con, $searchTitle) . "%'";
    }
    if ($searchUser !== '') {
        $searchIncome[] = "u.FullName LIKE '%" . mysqli_real_escape_string($con, $searchUser) . "%'";
        $searchExpense[] = "u.FullName LIKE '%" . mysqli_real_escape_string($con, $searchUser) . "%'";
    }
    if ($searchDetails !== '') {
        $searchIncome[] = "FundDetails LIKE '%" . mysqli_real_escape_string($con, $searchDetails) . "%'";
        $searchExpense[] = "ExpenseDetails LIKE '%" . mysqli_real_escape_string($con, $searchDetails) . "%'";
    }
} else {
    // Staff: restrict to own records
    $searchIncome[] = "tblfunds.RaisedBy = '$userId'";
    $searchExpense[] = "tblexpenses.UserID = '$userId'";
    // Staff cannot search by user
    if ($searchTitle !== '') {
        $searchIncome[] = "FundTitle LIKE '%" . mysqli_real_escape_string($con, $searchTitle) . "%'";
        $searchExpense[] = "ExpenseTitle LIKE '%" . mysqli_real_escape_string($con, $searchTitle) . "%'";
    }
    if ($searchDetails !== '') {
        $searchIncome[] = "FundDetails LIKE '%" . mysqli_real_escape_string($con, $searchDetails) . "%'";
        $searchExpense[] = "ExpenseDetails LIKE '%" . mysqli_real_escape_string($con, $searchDetails) . "%'";
    }
}
$searchIncomeSql = $searchIncome ? ' AND ' . implode(' AND ', $searchIncome) : '';
$searchExpenseSql = $searchExpense ? ' AND ' . implode(' AND ', $searchExpense) : '';

// --- DATA QUERIES (combine income and expense) ---
$unionParts = [];

if ($typeFilter == 'all' || $typeFilter == 'income') {
    $unionParts[] = "
        SELECT 
            'Income' AS Type,
            FundTitle AS Title,
            FundAmount AS Credit,
            NULL AS Debit,
            PaymentMode,
            FundDetails AS Details,
            u.FullName AS Staff,
            EntryDate AS CreatedAt,
            NULL AS VoucherImage
        FROM tblfunds
        LEFT JOIN tblusers u ON tblfunds.RaisedBy = u.ID
        $whereIncome" . ($searchIncomeSql ? str_replace('WHERE', 'AND', $searchIncomeSql) : '') . "
    ";
}

if ($typeFilter == 'all' || $typeFilter == 'expense') {
    $unionParts[] = "
        SELECT 
            'Expense' AS Type,
            ExpenseTitle AS Title,
            NULL AS Credit,
            ExpenseAmount AS Debit,
            PaymentMode,
            ExpenseDetails AS Details,
            u.FullName AS Staff,
            EntryDate AS CreatedAt,
            VoucherImage
        FROM tblexpenses
        LEFT JOIN tblusers u ON tblexpenses.UserID = u.ID
        $whereExpense" . ($searchExpenseSql ? str_replace('WHERE', 'AND', $searchExpenseSql) : '') . "
    ";
}

$unionQuery = implode(' UNION ALL ', $unionParts) . " $orderBy";
$records = mysqli_query($con, $unionQuery);

// --- TOTALS ---
$incomeWhereTotal = $whereIncome . ($searchIncomeSql ? str_replace('WHERE', 'AND', $searchIncomeSql) : '');
$expenseWhereTotal = $whereExpense . ($searchExpenseSql ? str_replace('WHERE', 'AND', $searchExpenseSql) : '');

$totalIncome = 0;
$totalExpense = 0;

if ($typeFilter == 'all' || $typeFilter == 'income') {
    $result = mysqli_query($con, "SELECT SUM(FundAmount) as total FROM tblfunds LEFT JOIN tblusers u ON tblfunds.RaisedBy = u.ID $incomeWhereTotal"); $row = $result ? mysqli_fetch_assoc($result) : null;
    $totalIncome = $row && $row['total'] ? $row['total'] : 0;
}

if ($typeFilter == 'all' || $typeFilter == 'expense') {
    $result = mysqli_query($con, "SELECT SUM(ExpenseAmount) as total FROM tblexpenses LEFT JOIN tblusers u ON tblexpenses.UserID = u.ID $expenseWhereTotal"); $row = $result ? mysqli_fetch_assoc($result) : null;
    $totalExpense = $row && $row['total'] ? $row['total'] : 0;
}

$availableFund = $totalIncome - $totalExpense;

// --- Opening Fund Calculation ---
$openingFrom = $from ? date('Y-m-d', strtotime($from . ' -1 day')) : null;
$openingIncome = 0;
$openingExpense = 0;
if ($openingFrom) {
    $openingIncomeWhere = "WHERE DATE(EntryDate) <= '$openingFrom'";
    $openingExpenseWhere = "WHERE DATE(EntryDate) <= '$openingFrom'";
    if ($userType != 'admin') {
        $openingIncomeWhere .= " AND tblfunds.RaisedBy = '$userId'";
        $openingExpenseWhere .= " AND tblexpenses.UserID = '$userId'";
    }
    $result = mysqli_query($con, "SELECT SUM(FundAmount) as total FROM tblfunds $openingIncomeWhere"); $row = $result ? mysqli_fetch_assoc($result) : null;
    $openingIncome = $row && $row['total'] ? $row['total'] : 0;
    $result = mysqli_query($con, "SELECT SUM(ExpenseAmount) as total FROM tblexpenses $openingExpenseWhere"); $row = $result ? mysqli_fetch_assoc($result) : null;
    $openingExpense = $row && $row['total'] ? $row['total'] : 0;
}
$openingFund = $openingIncome - $openingExpense;
$closingFund = $openingFund + $totalIncome - $totalExpense;

// --- Helper for sort links ---
function sortLink($label, $col, $currentSort, $currentOrder, $params) {
    $order = ($currentSort == $col && $currentOrder == 'ASC') ? 'desc' : 'asc';
    $icon = '';
    if ($currentSort == $col) {
        $icon = $currentOrder == 'ASC' ? ' <i class="fas fa-sort-up"></i>' : ' <i class="fas fa-sort-down"></i>';
    }
    $url = $_SERVER['PHP_SELF'].'?'.http_build_query(array_merge($params, ['sort'=>$col,'order'=>$order]));
    return "<a href=\"$url\" class=\"text-dark\">$label$icon</a>";
}
$params = $_GET;
unset($params['sort'], $params['order']);

// Get payment modes for filter dropdown
$paymentModes = [];
$paymentQuery = "SELECT DISTINCT PaymentMode FROM (
    SELECT PaymentMode FROM tblfunds WHERE PaymentMode IS NOT NULL AND PaymentMode != ''
    UNION
    SELECT PaymentMode FROM tblexpenses WHERE PaymentMode IS NOT NULL AND PaymentMode != ''
) AS combined ORDER BY PaymentMode";
$paymentResult = mysqli_query($con, $paymentQuery);
while($row = mysqli_fetch_assoc($paymentResult)) {
    $paymentModes[] = $row['PaymentMode'];
}

// --- Helper for Indian number formatting ---
function formatIndianCurrency($number) {
    $number = (float)$number;
    $negative = $number < 0;
    $number = abs($number);
    if ($number < 100000) {
        $result = number_format($number, 2);
    } else if ($number < 10000000) {
        $result = number_format($number / 100000, 2) . ' Lakh';
    } else {
        $result = number_format($number / 10000000, 2) . ' Crore';
    }
    return ($negative ? '-' : '') . $result;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Income & Expense Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 1rem; }
        .card-header { border-radius: 1rem 1rem 0 0; }
        .summary-card { background: linear-gradient(90deg, #43cea2 0%, #185a9d 100%); color: #fff; }
        .summary-value { font-size: 2rem; font-weight: bold; }
        .filter-btn { margin: 0.2rem 0.2rem 0.2rem 0; }
        .table th, .table td { vertical-align: middle !important; }
        .voucher-thumb { max-width: 60px; max-height: 60px; border-radius: 0.5rem; }
        .text-credit { color: #28a745; font-weight: bold; text-align: right; }
        .text-debit { color: #dc3545; font-weight: bold; text-align: right; }
        .text-amount { text-align: right; }
        .search-input {
            border-radius: 1.5rem;
            font-size: 1rem;
            padding: 0.5rem 1rem;
            margin-bottom: 0.5rem;
            width: 100%;
            border: 1px solid #43cea2;
        }
        @media (max-width: 767.98px) {
            .container, .container-fluid { padding-left: 0.5rem; padding-right: 0.5rem; }
            .card { margin-bottom: 1.2rem; }
            .table-responsive { font-size: 0.95rem; }
            .btn, .btn-block { font-size: 1rem; padding: 0.6rem 0.75rem; }
            .card-header h4, .card-header h5 { font-size: 1.1rem; }
            .search-input { font-size: 0.98rem; padding: 0.45rem 0.9rem; }
            .card-header form.form-row { flex-direction: column; }
            .card-header form.form-row > div { width: 100%; }
        }
    </style>
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // Auto submit for payment mode and type filter (dropdowns only)
        var filterSelects = document.querySelectorAll('.filter-select');
        filterSelects.forEach(function(select) {
            select.addEventListener('change', function() {
                var form = select.closest('form');
                if(form) form.submit();
            });
        });
    });
    </script>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-2 mb-2">
                   
                    <!-- Opening/Closing Fund Summary -->
                    <div class="row mb-4">
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #6a11cb 0%, #2575fc 100%);">
                                <div class="card-body">
                                    <div>Opening Fund</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($openingFund); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center">
                                <div class="card-body">
                                    <div>Total Income</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($totalIncome); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #f7971e 0%, #ffd200 100%); color: #333;">
                                <div class="card-body">
                                    <div>Total Expenses</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($totalExpense); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #43cea2 0%, #185a9d 100%);">
                                <div class="card-body">
                                    <div>Closing Fund</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($closingFund); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Filter Options -->
                    <div class="card shadow mb-4">
                        <div class="card-header bg-gradient-primary text-white">
                            <form class="form-inline flex-wrap" method="get" action="">
                                <div class="date-row mb-2">
                                    <button type="submit" name="filter" value="all" class="btn btn-info filter-btn <?php if($filter=='all') echo 'font-weight-bold'; ?>">All Time</button>
                                    <button type="submit" name="filter" value="today" class="btn btn-primary filter-btn <?php if($filter=='today') echo 'font-weight-bold'; ?>">Today</button>
                                    <button type="submit" name="filter" value="yesterday" class="btn btn-primary filter-btn <?php if($filter=='yesterday') echo 'font-weight-bold'; ?>">Yesterday</button>
                                    <button type="submit" name="filter" value="this_week" class="btn btn-primary filter-btn <?php if($filter=='this_week') echo 'font-weight-bold'; ?>">This Week</button>
                                    <button type="submit" name="filter" value="this_month" class="btn btn-primary filter-btn <?php if($filter=='this_month') echo 'font-weight-bold'; ?>">This Month</button>
                                    <button type="submit" name="filter" value="this_year" class="btn btn-primary filter-btn <?php if($filter=='this_year') echo 'font-weight-bold'; ?>">This Year</button>
                                    <span class="ml-2 mr-2 font-weight-bold">Custom:</span>
                                    <input type="date" name="from" class="form-control filter-btn" value="<?php echo isset($_GET['from']) ? $_GET['from'] : ''; ?>">
                                    <input type="date" name="to" class="form-control filter-btn" value="<?php echo isset($_GET['to']) ? $_GET['to'] : ''; ?>">
                                    <button type="submit" name="filter" value="custom" class="btn btn-success filter-btn <?php if($filter=='custom') echo 'font-weight-bold'; ?>">Apply</button>
                                </div>
                                <div class="filter-row mb-2 w-100">
                                    <div class="d-flex flex-wrap align-items-center">
                                        <select name="type_filter" class="form-control filter-btn filter-select">
                                            <option value="all" <?php if($typeFilter=='all') echo 'selected'; ?>>All Types</option>
                                            <option value="income" <?php if($typeFilter=='income') echo 'selected'; ?>>Income Only</option>
                                            <option value="expense" <?php if($typeFilter=='expense') echo 'selected'; ?>>Expense Only</option>
                                        </select>
                                        <select name="payment_mode" class="form-control filter-btn filter-select">
                                            <option value="">All Payment Modes</option>
                                            <?php foreach($paymentModes as $mode) { ?>
                                                <option value="<?php echo htmlspecialchars($mode); ?>" <?php if($paymentMode==$mode) echo 'selected'; ?>><?php echo htmlspecialchars($mode); ?></option>
                                            <?php } ?>
                                        </select>
                                        <input type="text" name="search_title" class="form-control filter-btn" placeholder="Search Title" value="<?php echo htmlspecialchars($searchTitle); ?>" style="max-width: 150px;">
                                        <?php if($userType == 'admin') { ?>
                                        <input type="text" name="search_user" class="form-control filter-btn" placeholder="Search User" value="<?php echo htmlspecialchars($searchUser); ?>" style="max-width: 150px;">
                                        <?php } ?>
                                        <input type="text" name="search_details" class="form-control filter-btn" placeholder="Search Details" value="<?php echo htmlspecialchars($searchDetails); ?>" style="max-width: 150px;">
                                        <button type="submit" class="btn btn-info filter-btn"><i class="fas fa-filter"></i> Filter</button>
                                        <a href="?<?php echo http_build_query(array_merge($_GET, ['export'=>'excel'])); ?>" class="btn btn-warning filter-btn ml-auto"><i class="fas fa-file-excel"></i> Export Excel</a>
                                    </div>
                                </div>
                                <!-- Preserve other filters -->
                                <?php foreach(['sort','order'] as $keep) if(isset($_GET[$keep])) echo '<input type="hidden" name="'.$keep.'" value="'.htmlspecialchars($_GET[$keep]).'">'; ?>
                            </form>
                        </div>
                    </div>
                 
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>