<?php
session_start();
include_once('includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

// --- FILTERS ---
function getDateRange($filter) {
    $today = date('Y-m-d');
    switch ($filter) {
        case 'today':
            return [$today, $today];
        case 'yesterday':
            $yesterday = date('Y-m-d', strtotime('-1 day'));
            return [$yesterday, $yesterday];
        case 'this_week':
            $start = date('Y-m-d', strtotime('monday this week'));
            $end = date('Y-m-d', strtotime('sunday this week'));
            return [$start, $end];
        case 'this_month':
            $start = date('Y-m-01');
            $end = date('Y-m-t');
            return [$start, $end];
        case 'this_year':
            $start = date('Y-01-01');
            $end = date('Y-12-31');
            return [$start, $end];
        case 'custom':
            $start = isset($_GET['from']) ? $_GET['from'] : $today;
            $end = isset($_GET['to']) ? $_GET['to'] : $today;
            return [$start, $end];
        default:
            return [null, null]; // all time
    }
}

// --- Helper function for Indian number formatting
function formatINR($number) {
    $number = (float)$number;
    $negative = $number < 0;
    $number = abs($number);
    $decimal = '';
    if (strpos($number, '.') !== false) {
        $parts = explode('.', $number);
        $number = $parts[0];
        $decimal = isset($parts[1]) && $parts[1] ? '.' . substr($parts[1] . '00', 0, 2) : '';
    }
    $len = strlen($number);
    if ($len > 3) {
        $last3 = substr($number, -3);
        $restUnits = substr($number, 0, $len - 3);
        $restUnits = preg_replace("/\B(?=(\d{2})+(?!\d))/", ",", $restUnits);
        $formatted = $restUnits . "," . $last3;
    } else {
        $formatted = $number;
    }
    return ($negative ? '-' : '') . $formatted . $decimal;
}

// --- Excel Export ---
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    $filter = isset($_GET['filter']) ? $_GET['filter'] : 'this_month';
    list($from, $to) = getDateRange($filter);
    $whereClause = "";
    if ($from && $to) {
        $whereClause = "WHERE DATE(t.TransactionDate) BETWEEN '$from' AND '$to'";
    }

    // Bank Account Filter
    $bankAccountId = isset($_GET['bank_account_id']) ? intval($_GET['bank_account_id']) : 0;
    if ($bankAccountId > 0) {
        $bankFilter = " AND t.BankAccountID = '$bankAccountId'";
        $whereClause = $whereClause ? $whereClause . $bankFilter : "WHERE t.BankAccountID = '$bankAccountId'";
    }

    // Transaction Type Filter
    $typeFilter = isset($_GET['type_filter']) ? $_GET['type_filter'] : 'all';
    if ($typeFilter != 'all') {
        $transTypeFilter = " AND t.TransactionType = '" . mysqli_real_escape_string($con, $typeFilter) . "'";
        $whereClause = $whereClause ? $whereClause . $transTypeFilter : "WHERE t.TransactionType = '" . mysqli_real_escape_string($con, $typeFilter) . "'";
    }

    // Opening balance for Excel export
    $exportOpeningBalance = 0;
    if ($filter != 'all' && $from) {
        $openingWhereClause = "WHERE DATE(t.TransactionDate) < '$from'";
        if ($bankAccountId > 0) {
            $openingWhereClause .= " AND t.BankAccountID = '$bankAccountId'";
        }
        $openingCreditQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
            $openingWhereClause AND t.TransactionType='Credit'";
        $_result = mysqli_query($con, $openingCreditQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
        $openingCredit = $row && $row['total'] ? $row['total'] : 0;

        $openingDebitQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
            $openingWhereClause AND t.TransactionType='Debit'";
        $_result = mysqli_query($con, $openingDebitQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
        $openingDebit = $row && $row['total'] ? $row['total'] : 0;

        $exportOpeningBalance = $openingCredit - $openingDebit;
    }

    // Totals for export (filtered period only)
    $totalWhereClause = $whereClause;
    $exportTotalCredit = 0;
    $exportTotalDebit = 0;

    $creditQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
        $totalWhereClause AND t.TransactionType='Credit'";
    $_result = mysqli_query($con, $creditQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
    $exportTotalCredit = $row && $row['total'] ? $row['total'] : 0;

    $debitQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
        $totalWhereClause AND t.TransactionType='Debit'";
    $_result = mysqli_query($con, $debitQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
    $exportTotalDebit = $row && $row['total'] ? $row['total'] : 0;

    $exportNetBalance = $exportTotalCredit - $exportTotalDebit;
    $exportClosingBalance = $exportOpeningBalance + $exportNetBalance;

    $orderBy = "ORDER BY t.TransactionDate DESC, t.CreatedDate DESC";

    $query = "SELECT t.*, b.BankName, b.AccountNumber, u.FullName 
        FROM tblaccounttransactions t 
        LEFT JOIN tblbankdetails b ON t.BankAccountID=b.ID 
        LEFT JOIN tblusers u ON t.AddedBy=u.ID 
        $whereClause $orderBy";

    $records = mysqli_query($con, $query);

    header("Content-Type: application/vnd.ms-excel");
    header("Content-Disposition: attachment; filename=account_transaction_report.xls");
    echo "Opening Balance: Rs " . formatINR($exportOpeningBalance) . "\n\n";
    echo "Bank\tAccount Number\tTransaction Type\tTitle\tCredit\tDebit\tReference\tDetails\tAdded By\tTransaction Date\n";
    while($row = mysqli_fetch_assoc($records)) {
        echo $row['BankName']."\t";
        echo $row['AccountNumber']."\t";
        echo $row['TransactionType']."\t";
        echo $row['TransactionTitle']."\t";
        echo ($row['TransactionType'] == 'Credit' ? $row['TransactionAmount'] : "")."\t";
        echo ($row['TransactionType'] == 'Debit' ? $row['TransactionAmount'] : "")."\t";
        echo ($row['ReferenceNumber'] ?: "")."\t";
        echo ($row['TransactionDetails'] ?: "")."\t";
        echo $row['FullName']."\t";
        echo $row['TransactionDate']."\n";
    }
    echo "\n\nSummary:\n";
    echo "Opening Balance\tRs " . formatINR($exportOpeningBalance) . "\n";
    echo "Total Credits\tRs " . formatINR($exportTotalCredit) . "\n";
    echo "Total Debits\tRs " . formatINR($exportTotalDebit) . "\n";
    echo "Net Balance\tRs " . formatINR($exportNetBalance) . "\n";
    echo "Closing Balance\tRs " . formatINR($exportClosingBalance) . "\n";
    exit();
}

// --- Default filter: this_month ---
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'this_month';
list($from, $to) = getDateRange($filter);

$whereClause = "";
if ($from && $to) {
    $whereClause = "WHERE DATE(t.TransactionDate) BETWEEN '$from' AND '$to'";
}

// Bank Account Filter
$bankAccountId = isset($_GET['bank_account_id']) ? intval($_GET['bank_account_id']) : 0;
if ($bankAccountId > 0) {
    $bankFilter = " AND t.BankAccountID = '$bankAccountId'";
    $whereClause = $whereClause ? $whereClause . $bankFilter : "WHERE t.BankAccountID = '$bankAccountId'";
}

// Transaction Type Filter
$typeFilter = isset($_GET['type_filter']) ? $_GET['type_filter'] : 'all';
if ($typeFilter != 'all') {
    $transTypeFilter = " AND t.TransactionType = '" . mysqli_real_escape_string($con, $typeFilter) . "'";
    $whereClause = $whereClause ? $whereClause . $transTypeFilter : "WHERE t.TransactionType = '" . mysqli_real_escape_string($con, $typeFilter) . "'";
}

// --- Sorting ---
$sortable = ['TransactionType','TransactionTitle','TransactionAmount','BankName','FullName','TransactionDate'];
$sort = isset($_GET['sort']) && in_array($_GET['sort'],$sortable) ? $_GET['sort'] : 'TransactionDate';
$order = (isset($_GET['order']) && strtolower($_GET['order'])=='asc') ? 'ASC' : 'DESC';

$orderBy = "ORDER BY t.$sort $order";

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// --- Search Filters ---
$searchTitle = isset($_GET['search_title']) ? trim($_GET['search_title']) : '';
$searchUser = isset($_GET['search_user']) ? trim($_GET['search_user']) : '';
$searchDetails = isset($_GET['search_details']) ? trim($_GET['search_details']) : '';

$searchFilters = [];
if ($searchTitle !== '') {
    $searchFilters[] = "t.TransactionTitle LIKE '%" . mysqli_real_escape_string($con, $searchTitle) . "%'";
}
if ($userType == 'admin' && $searchUser !== '') {
    $searchFilters[] = "u.FullName LIKE '%" . mysqli_real_escape_string($con, $searchUser) . "%'";
}
if ($searchDetails !== '') {
    $searchFilters[] = "t.TransactionDetails LIKE '%" . mysqli_real_escape_string($con, $searchDetails) . "%'";
}

$searchSql = $searchFilters ? ' AND ' . implode(' AND ', $searchFilters) : '';

// --- DATA QUERY ---
$query = "SELECT t.*, b.BankName, b.AccountNumber, u.FullName 
    FROM tblaccounttransactions t 
    LEFT JOIN tblbankdetails b ON t.BankAccountID=b.ID 
    LEFT JOIN tblusers u ON t.AddedBy=u.ID 
    $whereClause" . ($searchSql ? str_replace('WHERE', 'AND', $searchSql) : '') . " 
    $orderBy";

$records = mysqli_query($con, $query);

// --- TOTALS ---
$totalWhereClause = $whereClause . ($searchSql ? str_replace('WHERE', 'AND', $searchSql) : '');

$totalCredit = 0;
$totalDebit = 0;
$openingBalance = 0;

// Calculate Opening Balance (before the date range)
if ($filter != 'all' && $from) {
    $openingWhereClause = "WHERE DATE(t.TransactionDate) < '$from'";
    if ($bankAccountId > 0) {
        $openingWhereClause .= " AND t.BankAccountID = '$bankAccountId'";
    }
    $openingCreditQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
        $openingWhereClause AND t.TransactionType='Credit'";
    $_result = mysqli_query($con, $openingCreditQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
    $openingCredit = $row && $row['total'] ? $row['total'] : 0;

    $openingDebitQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
        $openingWhereClause AND t.TransactionType='Debit'";
    $_result = mysqli_query($con, $openingDebitQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
    $openingDebit = $row && $row['total'] ? $row['total'] : 0;

    $openingBalance = $openingCredit - $openingDebit;
}

// Totals for filtered period only
$creditQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
    LEFT JOIN tblusers u ON t.AddedBy=u.ID 
    $totalWhereClause AND t.TransactionType='Credit'";
$_result = mysqli_query($con, $creditQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
$totalCredit = $row && $row['total'] ? $row['total'] : 0;

$debitQuery = "SELECT SUM(t.TransactionAmount) as total FROM tblaccounttransactions t 
    LEFT JOIN tblusers u ON t.AddedBy=u.ID 
    $totalWhereClause AND t.TransactionType='Debit'";
$_result = mysqli_query($con, $debitQuery); $row = $_result ? mysqli_fetch_assoc($_result) : null;
$totalDebit = $row && $row['total'] ? $row['total'] : 0;

$netBalance = $totalCredit - $totalDebit;
$closingBalance = $openingBalance + $netBalance;

// --- Helper for sort links ---
function sortLink($label, $col, $currentSort, $currentOrder, $params) {
    $order = ($currentSort == $col && $currentOrder == 'ASC') ? 'desc' : 'asc';
    $icon = '';
    if ($currentSort == $col) {
        $icon = $currentOrder == 'ASC' ? ' <i class="fas fa-sort-up"></i>' : ' <i class="fas fa-sort-down"></i>';
    }
    $url = $_SERVER['PHP_SELF'].'?'.http_build_query(array_merge($params, ['sort'=>$col,'order'=>$order]));
    return "<a href=\"$url\" class=\"text-dark\">$label$icon</a>";
}
$params = $_GET;
unset($params['sort'], $params['order']);

// Get bank accounts for filter dropdown
$bankAccounts = [];
$bankQuery = "SELECT ID, BankName, AccountNumber FROM tblbankdetails WHERE IsActive=1 ORDER BY BankName";
$bankResult = mysqli_query($con, $bankQuery);
while($row = mysqli_fetch_assoc($bankResult)) {
    $bankAccounts[] = $row;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Ihita Industries | Account Transaction Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 1rem; box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075); }
        .card-header { border-radius: 1rem 1rem 0 0; }
        .summary-card { background: #f8f9fa; color: #495057; border: 1px solid #e9ecef; }
        .summary-value { font-size: 2rem; font-weight: bold; }
        .filter-btn { margin: 0.2rem 0.2rem 0.2rem 0; }
        .table th, .table td { vertical-align: middle !important; }
        .text-credit { color: #28a745; font-weight: bold; text-align: right; }
        .text-debit { color: #dc3545; font-weight: bold; text-align: right; }
        .text-amount { text-align: right; }
        .search-input {
            border-radius: 1.5rem;
            font-size: 1rem;
            padding: 0.5rem 1rem;
            margin-bottom: 0.5rem;
            width: 100%;
            border: 1px solid #ced4da;
        }
        .credit-card { background: linear-gradient(135deg, #e8f5e8 0%, #d4edda 100%); border-left: 4px solid #28a745; }
        .debit-card { background: linear-gradient(135deg, #fdf2f2 0%, #f8d7da 100%); border-left: 4px solid #dc3545; }
        .balance-card { background: linear-gradient(135deg, #e7f1ff 0%, #cce7ff 100%); border-left: 4px solid #0066cc; }
        .filter-header { background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); color: #495057; border-bottom: 2px solid #dee2e6; }
        .table-header { background: linear-gradient(135deg, #495057 0%, #6c757d 100%); color: white; }
        @media (max-width: 767.98px) {
            .container, .container-fluid { padding-left: 0.5rem; padding-right: 0.5rem; }
            .card { margin-bottom: 1.2rem; }
            .table-responsive { font-size: 0.95rem; }
            .btn, .btn-block { font-size: 1rem; padding: 0.6rem 0.75rem; }
            .card-header h4, .card-header h5 { font-size: 1.1rem; }
            .search-input { font-size: 0.98rem; padding: 0.45rem 0.9rem; }
            .card-header form.form-row { flex-direction: column; }
            .card-header form.form-row > div { width: 100%; }
        }
    </style>
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // Auto submit for filters (dropdowns only)
        var filterSelects = document.querySelectorAll('.filter-select');
        filterSelects.forEach(function(select) {
            select.addEventListener('change', function() {
                var form = select.closest('form');
                if(form) form.submit();
            });
        });
    });
    </script>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-2 mb-2">
                   
                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(135deg, #f0f8ff 0%, #ddeeff 100%); border-left: 4px solid #007bff;">
                                <div class="card-body">
                                    <div style="color: #007bff; font-weight: 600;">Opening Balance</div>
                                    <div class="summary-value" style="color: #007bff; font-size: 1.5rem;"><i class="fas fa-play-circle"></i> Rs.<?php echo formatINR($openingBalance); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card credit-card shadow text-center">
                                <div class="card-body">
                                    <div style="color: #28a745; font-weight: 600;">Total Credits</div>
                                    <div class="summary-value" style="color: #28a745; font-size: 1.5rem;"><i class="fas fa-plus-circle"></i> Rs.<?php echo formatINR($totalCredit); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card debit-card shadow text-center">
                                <div class="card-body">
                                    <div style="color: #dc3545; font-weight: 600;">Total Debits</div>
                                    <div class="summary-value" style="color: #dc3545; font-size: 1.5rem;"><i class="fas fa-minus-circle"></i> Rs.<?php echo formatINR($totalDebit); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(135deg, #f5f0ff 0%, #e6d9ff 100%); border-left: 4px solid #6f42c1;">
                                <div class="card-body">
                                    <div style="color: #6f42c1; font-weight: 600;">Closing Balance</div>
                                    <div class="summary-value" style="color: #6f42c1; font-size: 1.5rem;"><i class="fas fa-stop-circle"></i> Rs.<?php echo formatINR($closingBalance); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Filter Options -->
                    <div class="card shadow mb-4">
                        <div class="card-header filter-header">
                            <form class="form-inline flex-wrap" method="get" action="">
                                <div class="date-row mb-2">
                                     <button type="submit" name="filter" value="today" class="btn btn-outline-secondary filter-btn <?php if($filter=='today') echo 'font-weight-bold'; ?>">Today</button>
                                    <button type="submit" name="filter" value="yesterday" class="btn btn-outline-secondary filter-btn <?php if($filter=='yesterday') echo 'font-weight-bold'; ?>">Yesterday</button>
                                    <button type="submit" name="filter" value="this_week" class="btn btn-outline-secondary filter-btn <?php if($filter=='this_week') echo 'font-weight-bold'; ?>">This Week</button>
                                    <button type="submit" name="filter" value="this_month" class="btn btn-outline-secondary filter-btn <?php if($filter=='this_month') echo 'font-weight-bold'; ?>">This Month</button>
                                    <button type="submit" name="filter" value="this_year" class="btn btn-outline-secondary filter-btn <?php if($filter=='this_year') echo 'font-weight-bold'; ?>">This Year</button>
                                    <span class="ml-2 mr-2 font-weight-bold" style="color: #495057;">Custom:</span>
                                    <input type="date" name="from" class="form-control filter-btn" value="<?php echo isset($_GET['from']) ? $_GET['from'] : ''; ?>" style="border-color: #ced4da;">
                                    <input type="date" name="to" class="form-control filter-btn" value="<?php echo isset($_GET['to']) ? $_GET['to'] : ''; ?>" style="border-color: #ced4da;">
                                    <button type="submit" name="filter" value="custom" class="btn btn-primary filter-btn <?php if($filter=='custom') echo 'font-weight-bold'; ?>">Apply</button>
                                </div>
                                <div class="filter-row mb-2 w-100">
                                    <div class="d-flex flex-wrap align-items-center">
                                        <select name="bank_account_id" class="form-control filter-btn filter-select" style="border-color: #ced4da;">
                                            <option value="">All Bank Accounts</option>
                                            <?php foreach($bankAccounts as $bank) { ?>
                                                <option value="<?php echo $bank['ID']; ?>" <?php if($bankAccountId==$bank['ID']) echo 'selected'; ?>>
                                                    <?php echo htmlspecialchars($bank['BankName'] . ' - ' . $bank['AccountNumber']); ?>
                                                </option>
                                            <?php } ?>
                                        </select>
                                        <select name="type_filter" class="form-control filter-btn filter-select" style="border-color: #ced4da;">
                                            <option value="all" <?php if($typeFilter=='all') echo 'selected'; ?>>All Types</option>
                                            <option value="Credit" <?php if($typeFilter=='Credit') echo 'selected'; ?>>Credit Only</option>
                                            <option value="Debit" <?php if($typeFilter=='Debit') echo 'selected'; ?>>Debit Only</option>
                                        </select>
                                        <input type="text" name="search_title" class="form-control filter-btn" placeholder="Search Title" value="<?php echo htmlspecialchars($searchTitle); ?>" style="max-width: 150px; border-color: #ced4da;">
                                        <?php if($userType == 'admin') { ?>
                                        <input type="text" name="search_user" class="form-control filter-btn" placeholder="Search User" value="<?php echo htmlspecialchars($searchUser); ?>" style="max-width: 150px; border-color: #ced4da;">
                                        <?php } ?>
                                        <input type="text" name="search_details" class="form-control filter-btn" placeholder="Search Details" value="<?php echo htmlspecialchars($searchDetails); ?>" style="max-width: 150px; border-color: #ced4da;">
                                        <button type="submit" class="btn btn-info filter-btn"><i class="fas fa-filter"></i> Filter</button>
                                        <a href="?<?php echo http_build_query(array_merge($_GET, ['export'=>'excel'])); ?>" class="btn btn-success filter-btn ml-auto"><i class="fas fa-file-excel"></i> Export Excel</a>
                                    </div>
                                </div>
                                <!-- Preserve other filters -->
                                <?php foreach(['sort','order'] as $keep) if(isset($_GET[$keep])) echo '<input type="hidden" name="'.$keep.'" value="'.htmlspecialchars($_GET[$keep]).'">'; ?>
                            </form>
                        </div>
                    </div>

                    <!-- Account Transactions Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header table-header text-center p-2">
                            <h5 class="mb-0" style="font-size:1.1rem;">Account Transaction Report</h5>
                        </div>
                        <div class="card-body p-1">
                            <div class="table-responsive">
                            <table class="table table-bordered table-sm mb-0" style="font-size:0.98rem;">
                                <thead style="background-color: #f8f9fa; color: #495057;">
                                    <tr>
                                        <th style="min-width:30px; border-color: #dee2e6;">#</th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Date','TransactionDate',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Bank Account','BankName',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Type','TransactionType',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Title','TransactionTitle',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Credit','TransactionAmount',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Debit','TransactionAmount',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;">Reference</th>
                                        <th style="border-color: #dee2e6;">Details</th>
                                        <th style="border-color: #dee2e6;"><?php echo sortLink('Added By','FullName',$sort,$order,$params); ?></th>
                                        <th style="border-color: #dee2e6;">Created</th>
                                    </tr>
                                </thead>
                                <tbody>
                                <?php $i=1; while($row = mysqli_fetch_assoc($records)) { ?>
                                    <tr style="background-color: <?php echo ($i % 2 == 0) ? '#f8f9fa' : '#ffffff'; ?>;">
                                        <td style="border-color: #dee2e6;"><?php echo $i++; ?></td>
                                        <td style="border-color: #dee2e6;"><?php echo date('d-m-Y', strtotime($row['TransactionDate'])); ?></td>
                                        <td style="border-color: #dee2e6;">
                                            <small>
                                                <strong><?php echo htmlspecialchars($row['BankName']); ?></strong><br>
                                                <?php echo htmlspecialchars($row['AccountNumber']); ?>
                                            </small>
                                        </td>
                                        <td style="border-color: #dee2e6;">
                                            <?php if($row['TransactionType'] == 'Credit') { ?>
                                                <span class="badge" style="background-color: #28a745; color: white;">Credit</span>
                                            <?php } else { ?>
                                                <span class="badge" style="background-color: #dc3545; color: white;">Debit</span>
                                            <?php } ?>
                                        </td>
                                        <td style="border-color: #dee2e6;"><?php echo htmlspecialchars($row['TransactionTitle']); ?></td>
                                        <td class="text-credit" style="border-color: #dee2e6;">
                                            <?php echo ($row['TransactionType'] == 'Credit') ? formatINR($row['TransactionAmount']) : ''; ?>
                                        </td>
                                        <td class="text-debit" style="border-color: #dee2e6;">
                                            <?php echo ($row['TransactionType'] == 'Debit') ? formatINR($row['TransactionAmount']) : ''; ?>
                                        </td>
                                        <td style="border-color: #dee2e6;"><?php echo htmlspecialchars($row['ReferenceNumber'] ?: '-'); ?></td>
                                        <td style="border-color: #dee2e6;"><?php echo htmlspecialchars($row['TransactionDetails'] ?: '-'); ?></td>
                                        <td style="border-color: #dee2e6;"><?php echo htmlspecialchars($row['FullName']); ?></td>
                                        <td style="border-color: #dee2e6;"><?php echo date('d-m-Y H:i', strtotime($row['CreatedDate'])); ?></td>
                                    </tr>
                                <?php } ?>
                                </tbody>
                            </table>
                            </div>
                        </div>
                    </div>
                 
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>
